<?php

namespace Drupal\duma\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\Plugin\Field\FieldFormatter\EntityReferenceFormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\media\Entity\Media;
use Drupal\file\Entity\File;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Component\Utility\Html;

/**
 * Plugin implementation of the 'duma_audio_player' formatter.
 *
 * @FieldFormatter(
 *   id = "duma_audio_player",
 *   label = @Translation("DUMA Audio Player"),
 *   description = @Translation("Display audio files with HTML5 audio player"),
 *   field_types = {
 *     "entity_reference"
 *   }
 * )
 */
class DumaFormatter extends EntityReferenceFormatterBase implements ContainerFactoryPluginInterface {

  /**
   * The file URL generator.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected $fileUrlGenerator;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a DumaFormatter object.
   *
   * @param string $plugin_id
   *   The plugin_id for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the formatter is associated.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The formatter label display setting.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\Core\File\FileUrlGeneratorInterface $file_url_generator
   *   The file URL generator.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    $label,
    $view_mode,
    array $third_party_settings,
    FileUrlGeneratorInterface $file_url_generator,
    ConfigFactoryInterface $config_factory
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    $this->fileUrlGenerator = $file_url_generator;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('file_url_generator'),
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'show_filename' => TRUE,
      'show_filesize' => TRUE,
      'show_download_link' => FALSE,
      'autoplay' => FALSE,
      'loop' => FALSE,
      'preload' => 'metadata',
      'controls_list' => '',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = parent::settingsForm($form, $form_state);

    $elements['show_filename'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show filename'),
      '#default_value' => $this->getSetting('show_filename'),
    ];

    $elements['show_filesize'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show file size'),
      '#default_value' => $this->getSetting('show_filesize'),
    ];

    $elements['show_download_link'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show download link'),
      '#default_value' => $this->getSetting('show_download_link'),
      '#description' => $this->t('Display a download link below the audio player.'),
    ];

    $elements['autoplay'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Autoplay'),
      '#default_value' => $this->getSetting('autoplay'),
      '#description' => $this->t('Note: Most browsers block autoplay unless the user has interacted with the site.'),
    ];

    $elements['loop'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Loop playback'),
      '#default_value' => $this->getSetting('loop'),
    ];

    $elements['preload'] = [
      '#type' => 'select',
      '#title' => $this->t('Preload'),
      '#options' => [
        'none' => $this->t('None - Do not preload'),
        'metadata' => $this->t('Metadata - Preload metadata only'),
        'auto' => $this->t('Auto - Let browser decide'),
      ],
      '#default_value' => $this->getSetting('preload'),
      '#description' => $this->t('Hint to the browser about how much buffering is advisable.'),
    ];

    $elements['controls_list'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Control list'),
      '#default_value' => $this->getSetting('controls_list'),
      '#description' => $this->t('Space-separated list of controls to hide. Options: nodownload, nofullscreen, noremoteplayback. Example: "nodownload nofullscreen"'),
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];

    if ($this->getSetting('show_filename')) {
      $summary[] = $this->t('Show filename');
    }

    if ($this->getSetting('show_filesize')) {
      $summary[] = $this->t('Show file size');
    }

    if ($this->getSetting('show_download_link')) {
      $summary[] = $this->t('Show download link');
    }

    if ($this->getSetting('autoplay')) {
      $summary[] = $this->t('Autoplay enabled');
    }

    if ($this->getSetting('loop')) {
      $summary[] = $this->t('Loop playback');
    }

    $preload = $this->getSetting('preload');
    $summary[] = $this->t('Preload: @preload', ['@preload' => $preload]);

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];
    $source_field = $this->configFactory->get('duma.settings')->get('source_field') ?? 'field_media_audio_file';

    foreach ($items as $delta => $item) {
      $media = Media::load($item->target_id);
      
      if (!$media) {
        continue;
      }

      // Get the file from the media entity
      if (!$media->hasField($source_field) || $media->get($source_field)->isEmpty()) {
        continue;
      }

      $file_entity = $media->get($source_field)->entity;
      
      if (!$file_entity instanceof File) {
        continue;
      }

      $file_uri = $file_entity->getFileUri();
      $filename = $file_entity->getFilename();
      $filesize = $file_entity->getSize();
      $preview_url = $this->fileUrlGenerator->generateString($file_uri);

      // Build audio attributes
      $audio_attributes = [
        'controls' => 'controls',
        'src' => $preview_url,
        'class' => ['duma-audio-element'],
      ];

      if ($this->getSetting('autoplay')) {
        $audio_attributes['autoplay'] = 'autoplay';
      }

      if ($this->getSetting('loop')) {
        $audio_attributes['loop'] = 'loop';
      }

      $preload = $this->getSetting('preload');
      if ($preload) {
        $audio_attributes['preload'] = $preload;
      }

      $controls_list = trim($this->getSetting('controls_list'));
      if ($controls_list) {
        $audio_attributes['controlslist'] = $controls_list;
      }

      // Build the render array
      $elements[$delta] = [
        '#theme' => 'duma_audio_formatter',
        '#audio_url' => $preview_url,
        '#audio_attributes' => $audio_attributes,
        '#filename' => $filename,
        '#filesize' => format_size($filesize),
        '#show_filename' => $this->getSetting('show_filename'),
        '#show_filesize' => $this->getSetting('show_filesize'),
        '#show_download_link' => $this->getSetting('show_download_link'),
        '#cache' => [
          'tags' => $file_entity->getCacheTags(),
        ],
      ];
    }

    // Attach the library for consistent styling
    if (!empty($elements)) {
      $elements['#attached']['library'][] = 'duma/duma_formatter';
    }

    return $elements;
  }

}