<?php

namespace Drupal\duma\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Url;
use Drupal\media\Entity\Media;
use Drupal\file\Entity\File;
use Drupal\Core\Access\CsrfTokenGenerator;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Component\Utility\Html;
use Drupal\Core\StringTranslation\ByteSizeMarkup;


/**
 * Plugin implementation of the 'duma_widget' widget.
 *
 * @FieldWidget(
 *   id = "duma_widget",
 *   label = @Translation("Direct Upload Media Audio"),
 *   field_types = {
 *     "entity_reference"
 *   }
 * )
 */
class DumaWidget extends WidgetBase implements ContainerFactoryPluginInterface {

  protected $messenger;
  protected $csrfTokenGenerator;
  protected $fileUrlGenerator;
  protected $loggerFactory;
  protected $configFactory;
  protected $entityTypeManager;
  protected $currentUser;
  
  /**
   * Constructs a DumaWidget object.
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    array $third_party_settings,
    MessengerInterface $messenger,
    CsrfTokenGenerator $csrf_token_generator,
    FileUrlGeneratorInterface $file_url_generator,
    LoggerChannelFactoryInterface $logger_factory,
    ConfigFactoryInterface $config_factory,
    EntityTypeManagerInterface $entity_type_manager,
    AccountProxyInterface $current_user
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
    $this->messenger = $messenger;
    $this->csrfTokenGenerator = $csrf_token_generator;
    $this->fileUrlGenerator = $file_url_generator;
    $this->loggerFactory = $logger_factory;
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('messenger'),
      $container->get('csrf_token'),
      $container->get('file_url_generator'),
      $container->get('logger.factory'),
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('current_user')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'max_size' => '52428800', // 50MB default
      'max_files' => 1,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $element = parent::settingsForm($form, $form_state);

    $element['max_size'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Maximum file size'),
      '#default_value' => $this->getSetting('max_size'),
      '#description' => $this->t('Maximum file size in bytes. Leave empty for no limit.'),
      '#size' => 20,
    ];

    $element['max_files'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum number of files'),
      '#default_value' => $this->getSetting('max_files'),
      '#description' => $this->t('Maximum number of audio files that can be uploaded.'),
      '#min' => 1,
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    
    $max_size = $this->getSetting('max_size');
    if (!empty($max_size)) {
      $size_int = (int) $max_size;
      $size_markup = ByteSizeMarkup::create($size_int);
      $summary[] = $this->t('Maximum file size: @size', [
        '@size' => $size_markup,
      ]);
    }
    
    $max_files = $this->getSetting('max_files');
    $summary[] = $this->t('Maximum files: @count', [
      '@count' => $max_files,
    ]);

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    // 1. Get the Media entity ID (target_id)
    $target_id = $items[$delta]->target_id ?? NULL;
    $preview_url = '';
    $filename = '';
    $filesize = '';
    
    // 2. Load the Media entity and determine the current state.
    if (!empty($target_id) && $media = Media::load($target_id)) {
      $source_field = $this->configFactory->get('duma.settings')->get('source_field') ?? 'field_media_audio_file'; 
      
      if ($media->hasField($source_field) && !$media->get($source_field)->isEmpty()) {
        $file_entity = $media->get($source_field)->entity;
        
        if ($file_entity instanceof File) {
          $file_uri = $file_entity->getFileUri();
          $filename = $file_entity->getFilename();
          $size_value = $file_entity->getSize() ?? 0;
          $filesize_markup = ByteSizeMarkup::create($size_value);
          $filesize = (string) $filesize_markup;
          
          // Generate the direct file URL for the HTML5 <audio> tag.
          $preview_url = $this->fileUrlGenerator->generateString($file_uri);
        }
      }
    }

    // Generate a consistent, predictable ID for the file input
    $field_name = $items->getName();
    $input_id = Html::getId('duma-file-input-' . $field_name . '-' . $delta);

    // 3. Main wrapper and hidden input.
    $element['upload_container'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => [
          'duma-upload-container', 
          'duma-delta-' . $delta,
          $delta === 0 ? 'duma-delta-first duma-delta-0' : '',
          $delta > 0 && empty($target_id) ? 'duma-hidden-slot' : '',
          $this->getSetting('max_files') > 1 ? 'duma-multi-enabled' : '',
          empty($target_id) ? 'duma-slot-default' : 'duma-slot-in-use',
        ],
        'data-delta' => $delta,
      ],
      '#weight' => 0,
    ];
    
    // Hidden target_id input
    $element['upload_container']['target_id'] = [
      '#type' => 'hidden',
      '#default_value' => $target_id,
      '#attributes' => [
        'class' => ['duma-media-id-input'],
      ],
    ];

    // 4. CSRF Token and Hidden Data
    $upload_url = Url::fromRoute('duma.upload_file')->toString();
    $token = $this->csrfTokenGenerator->get('duma_upload_token');

    $element['upload_container']['token_url_data'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['duma-token-url-data'],
        'data-upload-url' => $upload_url,
        'data-csrf-token' => $token,
        'data-max-size' => (string) ($this->getSetting('max_size') ?? '0'),
        'data-preview-style' => '',
      ],
    ];

    // 5. Preview Area (HTML5 Audio Player) - FIXED: Use container type with proper wrapper
    $element['upload_container']['preview'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['duma-upload-preview'],
      ],
    ];
    
    // Add the preview markup as a child element if we have content
    if (!empty($preview_url)) {
      $element['upload_container']['preview']['content'] = [
        '#markup' => $this->getPreviewMarkup($preview_url, $filename, $filesize),
      ];
    }

    // 6. File Input and Label
    $element['upload_container']['label'] = [
      '#type' => 'html_tag',
      '#tag' => 'label',
      '#attributes' => [
        'class' => ['duma-upload-label'],
        'for' => $input_id,
      ],
      '#value' => $this->getDefaultLabelContent($preview_url),
    ];

    $element['upload_container']['file_input'] = [
      '#type' => 'file',
      '#title' => $this->t('Upload Audio File'),
      '#title_display' => 'invisible',
      '#id' => $input_id,
      '#attributes' => [
        'class' => ['duma-upload-file-input'],
        'accept' => '.mp3,.wav,.ogg,.m4a,.opus,.flac,.aac',
      ],
    ];
    
    // 7. Actions Container (Remove Button and Error Messages)
    $element['upload_container']['actions'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => [
          'duma-upload-actions',
          empty($target_id) ? 'duma-hidden-actions' : '',
        ],
      ],
    ];

    // Remove Button
    $element['upload_container']['actions']['remove_button'] = [
      '#type' => 'html_tag',
      '#tag' => 'button',
      '#attributes' => [
        'type' => 'button',
        'class' => ['duma-remove-button', 'button', 'button--danger'],
      ],
      '#value' => $this->t('Remove'),
    ];
    
    // Error/Status Message Container
    $element['upload_container']['actions']['message'] = [
      '#type' => 'markup',
      '#markup' => '<div class="duma-error-message"></div>',
      '#weight' => 10,
    ];
    
    // 8. Attach Library
    $element['upload_container']['#attached']['library'][] = 'duma/duma';
    
    return $element;
  }

  /**
   * Generates the markup for the audio file preview (HTML5 <audio> tag).
   */
  protected function getPreviewMarkup(string $preview_url, string $filename, string $filesize): string {
    if (empty($preview_url)) {
      return '';
    }

    $safe_url = Html::escape($preview_url);
    $safe_filename = Html::escape($filename);
    $safe_filesize = Html::escape($filesize);

    $output = '<div class="duma-audio-wrapper">';
    $output .= '<audio controls src="' . $safe_url . '">' . $this->t('Your browser does not support the audio element.') . '</audio>';
    $output .= '<div class="duma-audio-details">';
    $output .= '<div class="duma-audio-filename">' . $safe_filename . '</div>';
    $output .= '<div class="duma-audio-filesize">(' . $safe_filesize . ')</div>';
    $output .= '</div>';
    $output .= '</div>';

    return $output;
  }

  /**
   * Gets the content for the default upload label.
   */
  protected function getDefaultLabelContent(string $preview_url): string {
    if (!empty($preview_url)) {
      return (string) $this->t('Click to replace audio file');
    }

    return (string) $this->t('Click to upload audio file');
  }

  /**
   * Massages the form values into the correct format for saving.
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    $massaged_values = [];
    
    foreach ($values as $delta => $value) {
      $target_id = NULL;
      
      if (isset($value['upload_container']['target_id'])) {
        $target_id = $value['upload_container']['target_id'];
      }
      elseif (isset($value['target_id'])) {
        $target_id = $value['target_id'];
      }
      
      if (!empty($target_id)) {
        $target_id = (int) $target_id;
        
        $media = Media::load($target_id);
        if ($media) {
          $massaged_values[] = [
            'target_id' => $target_id,
          ];
        }
      }
    }
    
    return $massaged_values;
  }

}