# DUMA - Direct Upload Media Audio

A Drupal 11 custom module that provides a streamlined audio upload widget for entity reference fields. DUMA allows users to upload audio files directly from forms with instant preview and playback capabilities.

## Features

- **Direct Audio Upload**: Upload audio files directly without leaving the form
- **Instant Preview**: HTML5 audio player with immediate playback after upload
- **Multiple Audio Formats**: Supports MP3, WAV, OGG, Opus, M4A, FLAC, AAC, and WebM
- **Multi-value Support**: Handle multiple audio uploads with auto-expanding slots
- **File Size Validation**: Configurable maximum file size limits
- **Rate Limiting**: Prevent abuse with per-user upload rate limits
- **Custom Labels**: Upload custom label images per content type
- **Drag-and-Drop Ready**: Compatible with Drupal's tabledrag for field ordering
- **Comprehensive Logging**: Track uploads and errors for debugging

## Requirements

- Drupal 11.x
- Media module (core)
- File module (core)
- An audio media type configured on your site

## Installation

1. Download or clone this module to your `modules/custom` directory
2. Enable the module:
   ```bash
   drush en duma
   ```
3. Clear caches:
   ```bash
   drush cr
   ```

## Configuration

### Initial Setup

1. Navigate to **Configuration → Media → DUMA Settings** (`/admin/config/media/duma`)

2. Configure the following settings:

   **General Settings:**
   - **Media Type**: Machine name of your audio media type (default: `audio`)
   - **Source Field**: Machine name of the file field on your media type (default: `field_media_audio_file`)

   **File Upload Settings:**
   - **Maximum File Size**: Maximum upload size in MB (default: 50 MB, max: 500 MB)
   - **Maximum Uploads Per Minute**: Rate limit per user (default: 60, max: 1000)

3. Click **Save configuration**

### Adding DUMA Widget to a Field

1. Go to **Structure → Content types** and select your content type
2. Navigate to **Manage form display**
3. Find your entity reference field that references media entities
4. Change the widget to **Direct Upload Media Audio**
5. Configure widget settings:
   - Maximum file size (specific to this field)
   - Maximum number of files (for multi-value fields)
6. Save the form display

### Custom Label Images (Optional)

Upload custom label images for each content type to replace the default "Click to upload audio file" text:

1. Go to **DUMA Settings** (`/admin/config/media/duma`)
2. Scroll to **Custom Widget Labels** section
3. Upload an image for each content type (PNG or SVG recommended)
4. Maximum size matches your configured file size limit
5. Save configuration

Custom labels will appear in the widget instead of default text.

## Usage

### For Content Editors

**Uploading an Audio File:**

1. Navigate to the content creation/edit form
2. Find the audio field with the DUMA widget
3. Click on the upload area or label
4. Select an audio file from your device
5. Wait for upload to complete
6. Audio player appears with instant playback capability

**Replacing an Audio File:**

1. Click "Click to replace audio file" button
2. Select a new audio file
3. Previous file is automatically replaced

**Removing an Audio File:**

1. Click the **Remove** button below the audio player
2. Audio file is removed from the field
3. Field returns to empty state

**Multi-value Fields:**

- First upload slot is always visible
- Additional slots appear automatically after uploading
- Remove non-first slots to hide them again

### For Developers

**Creating a Custom Audio Media Type:**

```yaml
# config/install/media.type.audio.yml
langcode: en
status: true
dependencies: {  }
id: audio
label: Audio
description: 'Audio files'
source: audio_file
queue_thumbnail_downloads: false
new_revision: false
source_configuration:
  source_field: field_media_audio_file
field_map: {  }
```

**Programmatic Configuration:**

```php
// Get DUMA config
$config = \Drupal::config('duma.settings');

// Update settings
\Drupal::configFactory()->getEditable('duma.settings')
  ->set('media_type', 'custom_audio')
  ->set('source_field', 'field_custom_audio_file')
  ->set('max_size', '104857600') // 100MB in bytes
  ->set('max_uploads_per_minute', 30)
  ->save();
```

**Accessing Uploaded Media:**

```php
// In a custom module or theme
$node = \Drupal\node\Entity\Node::load($nid);
$media_items = $node->get('field_audio')->referencedEntities();

foreach ($media_items as $media) {
  $source_field = 'field_media_audio_file';
  if ($media->hasField($source_field)) {
    $file = $media->get($source_field)->entity;
    if ($file) {
      $url = $file->createFileUrl();
      $filename = $file->getFilename();
      $filesize = $file->getSize();
    }
  }
}
```

## File Structure

```
duma/
├── README.md
├── duma.info.yml
├── duma.libraries.yml
├── duma.permissions.yml
├── duma.routing.yml
├── css/
│   └── duma.css
├── js/
│   └── duma.js
└── src/
    ├── Controller/
    │   └── DumaController.php
    ├── Form/
    │   └── DumaSettingsForm.php
    └── Plugin/
        └── Field/
            └── FieldWidget/
                └── DumaWidget.php
```

## Permissions

- **Upload DUMA files**: Required to upload audio files through the widget (assigned to authenticated users by default)
- **Administer site configuration**: Required to access DUMA settings page

## Supported File Formats

### Audio Formats
- MP3 (`.mp3`) - `audio/mpeg`
- WAV (`.wav`) - `audio/wav`, `audio/wave`, `audio/x-wav`
- OGG (`.ogg`) - `audio/ogg`, `application/ogg`
- Opus (`.opus`) - `audio/opus`, `application/ogg`
- M4A (`.m4a`) - `audio/mp4`, `audio/x-m4a`
- FLAC (`.flac`) - `audio/flac`, `audio/x-flac`
- AAC (`.aac`) - `audio/aac`, `audio/aacp`
- WebM (`.webm`) - `audio/webm`

### Image Formats (for custom labels)
- JPG/JPEG (`.jpg`, `.jpeg`)
- PNG (`.png`)
- GIF (`.gif`)
- SVG (`.svg`)

## Troubleshooting

### Upload fails with "No file was uploaded"
- Check that JavaScript is enabled
- Verify CSRF token is being generated correctly
- Check browser console for JavaScript errors

### Upload fails with "File type not allowed"
- Verify the file extension is in the supported list
- Check server MIME type detection (some servers detect `.opus` as `application/octet-stream`)
- Review watchdog logs: `drush wd-show --type=duma`

### Audio player doesn't appear after upload
- Clear Drupal caches: `drush cr`
- Check that the media type has the correct source field configured
- Verify file permissions in `sites/default/files/duma_uploads/`

### Rate limit errors
- Increase the "Maximum Uploads Per Minute" setting
- Clear user's tempstore: It resets automatically after 60 seconds

### Preview not showing on page reload
- Ensure the media entity was saved correctly
- Check that the file entity exists and is accessible
- Verify the source field name matches your media type configuration

## Logging

DUMA logs important events to the Drupal watchdog:

- File uploads (info level)
- Upload errors (error level)
- Configuration changes (info level)
- Rate limit violations (warning level)
- MIME type rejections (warning level)

View logs:
```bash
drush wd-show --type=duma
```

## Security Considerations

- Files are validated by both MIME type and extension
- CSRF tokens protect upload endpoint
- Rate limiting prevents abuse
- File size limits prevent DoS attacks
- Only authenticated users can upload by default
- Uploaded files are placed in public directory - consider private file system for sensitive audio

## Performance Tips

- Set reasonable file size limits (50MB default is good for most cases)
- Enable Drupal's file aggregation for CSS/JS
- Consider using a CDN for serving uploaded audio files
- For high-traffic sites, increase rate limits or implement queue-based processing

## Credits

**Module Name**: DUMA (Direct Upload Media Audio)  
**Drupal Version**: 11.x  
**License**: GPL-2.0-or-later

## Support

For bug reports and feature requests, please contact your site administrator or module maintainer.

## Changelog

### Version 1.0.0
- Initial release
- Direct audio upload widget
- Multi-value field support
- Custom label images per content type
- Rate limiting
- Comprehensive settings form
- HTML5 audio preview
- Support for all major audio formats