<?php

namespace Drupal\dumi\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\JsonResponse;
use Drupal\Core\Flood\FloodInterface;
use Drupal\media\Entity\Media;
use Drupal\file\Entity\File;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\file\FileUsage\FileUsageInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Access\CsrfTokenGenerator;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Drupal\image\Entity\ImageStyle;

/**
 * Controller for file uploads with multi-upload support.
 */
class DumiController extends ControllerBase {

  protected $currentUser;
  protected $fileUsage;
  protected $loggerFactory;
  protected $configFactory;
  protected $entityTypeManager;
  protected $csrfToken;
  protected $fileSystem;
  protected $fileUrlGenerator;
  protected $flood;

  public function __construct(
    AccountProxyInterface $current_user,
    FileUsageInterface $file_usage,
    LoggerChannelFactoryInterface $logger_factory,
    ConfigFactoryInterface $config_factory,
    EntityTypeManagerInterface $entity_type_manager,
    CsrfTokenGenerator $csrf_token,
    FileSystemInterface $file_system,
    FileUrlGeneratorInterface $file_url_generator,
    FloodInterface $flood
  ) {
    $this->currentUser = $current_user;
    $this->fileUsage = $file_usage;
    $this->loggerFactory = $logger_factory;
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->csrfToken = $csrf_token;
    $this->fileSystem = $file_system;
    $this->fileUrlGenerator = $file_url_generator;
    $this->flood = $flood;
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('current_user'),
      $container->get('file.usage'),
      $container->get('logger.factory'),
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('csrf_token'),
      $container->get('file_system'),
      $container->get('file_url_generator'),
      $container->get('flood')
    );
  }

  /**
   * Handles the file upload from the widget.
   */
  public function uploadFile(Request $request) {
    $logger = $this->loggerFactory->get('dumi');
    
    // SECURITY: Authentication check
    if (!$this->currentUser->isAuthenticated()) {
      $logger->warning('Anonymous user attempted file upload.');
      return new JsonResponse(['error' => 'Authentication required to upload files.'], 401);
    }

    // SECURITY: Permission check
    if (!$this->currentUser->hasPermission('upload dumi files')) {
      $logger->warning('User @uid attempted unauthorized file upload.', ['@uid' => $this->currentUser->id()]);
      return new JsonResponse(['error' => 'You do not have permission to upload files.'], 403);
    }
    
    // SECURITY: CSRF token validation
    $request_token = $request->headers->get('X-CSRF-Token');
    if (!$request_token || !$this->csrfToken->validate($request_token, 'dumi_upload')) {
      $logger->error('Invalid CSRF token for user @uid.', ['@uid' => $this->currentUser->id()]);
      return new JsonResponse(['error' => 'A security error occurred. Please refresh the page and try again.'], 403);
    }
    
    // Rate limiting using Drupal's Flood service
    // This works across multiple servers and with various cache backends
    $flood_identifier = 'dumi.upload.' . $this->currentUser->id();
    $flood_window = 3600; // 1 hour
    $flood_threshold = 50; // 50 uploads per hour per user
    
    if (!$this->flood->isAllowed('dumi_upload', $flood_threshold, $flood_window, $flood_identifier)) {
      $logger->warning('User @uid exceeded upload rate limit (@threshold per hour)', [
        '@uid' => $this->currentUser->id(),
        '@threshold' => $flood_threshold,
      ]);
      return new JsonResponse([
        'error' => 'Upload rate limit exceeded. Please wait before uploading more files.'
      ], 429);
    }
    
    // Register this upload attempt
    $this->flood->register('dumi_upload', $flood_window, $flood_identifier);
    
    $file_input_name = 'dumi_upload';
    
    try {
      $uploaded_file = $request->files->get($file_input_name);
      
      if (!$uploaded_file || !$uploaded_file instanceof UploadedFile) {
        $logger->error('No file was found in the upload request.');
        return new JsonResponse(['error' => 'No file was received. Please try again.'], 400);
      }

      if (!$uploaded_file->isValid()) {
        $logger->error('Invalid file upload: ' . $uploaded_file->getErrorMessage());
        return new JsonResponse(['error' => 'File upload failed: ' . $uploaded_file->getErrorMessage()], 400);
      }

      // Get configuration
      $config = $this->configFactory->get('dumi.settings');
      $max_size = (int) $config->get('file_size_limit') ?: 5;
      $max_size_bytes = $max_size * 1024 * 1024;

      $file_size = $uploaded_file->getSize();
      $file_mime = $uploaded_file->getMimeType();
      $original_name = $uploaded_file->getClientOriginalName();
      
      // SECURITY: File size validation
      if ($file_size > $max_size_bytes) {
        return new JsonResponse([
          'error' => "File size exceeds the limit of {$max_size} MB."
        ], 400);
      }

      // SECURITY: Extension validation
      $allowed_extensions = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
      $file_extension = strtolower($uploaded_file->getClientOriginalExtension());
      if (!in_array($file_extension, $allowed_extensions)) {
        return new JsonResponse([
          'error' => 'Only JPG, JPEG, PNG, GIF, and WebP files are allowed.'
        ], 400);
      }

      // SECURITY: MIME type validation with fallback handling
      // Note: Some browsers/systems send application/octet-stream for images
      $allowed_mimes = [
        'image/jpeg',
        'image/jpg', 
        'image/png',
        'image/gif',
        'image/webp',
        'application/octet-stream', // Fallback MIME type from some browsers
      ];
      
      if (!in_array($file_mime, $allowed_mimes)) {
        $logger->warning('File upload blocked due to MIME type mismatch. Extension: @ext, MIME: @mime', [
          '@ext' => $file_extension,
          '@mime' => $file_mime,
        ]);
        return new JsonResponse([
          'error' => 'File type not allowed. Please upload a valid image file.'
        ], 400);
      }
      
      // Additional validation: If MIME is octet-stream, verify it's actually an image
      if ($file_mime === 'application/octet-stream') {
        $temp_path = $uploaded_file->getRealPath();
        $image_info = @getimagesize($temp_path);
        
        if ($image_info === FALSE) {
          $logger->warning('File rejected: claimed to be image but getimagesize failed. Extension: @ext', [
            '@ext' => $file_extension,
          ]);
          return new JsonResponse([
            'error' => 'File does not appear to be a valid image.'
          ], 400);
        }
        
        // Update MIME type to actual image MIME type detected
        $file_mime = $image_info['mime'];
        $logger->info('Corrected MIME type from octet-stream to @mime for file with extension @ext', [
          '@mime' => $file_mime,
          '@ext' => $file_extension,
        ]);
      }

      // Sanitize filename
      $safe_filename = preg_replace('/[^a-zA-Z0-9_\-\.]/', '_', $original_name);
      $safe_filename = preg_replace('/_+/', '_', $safe_filename);
      
      // Prepare destination directory
      $destination_dir = 'public://dumi';
      if (!$this->fileSystem->prepareDirectory($destination_dir, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS)) {
        throw new \Exception('Failed to prepare destination directory');
      }

      // Save file
      $temp_path = $uploaded_file->getRealPath();
      $file_data = file_get_contents($temp_path);
      
      if ($file_data === FALSE) {
        throw new \Exception('Failed to read uploaded file');
      }
      
      $destination_uri = $this->fileSystem->getDestinationFilename($destination_dir . '/' . $safe_filename, FileSystemInterface::EXISTS_RENAME);
      $final_uri = $this->fileSystem->saveData($file_data, $destination_uri, FileSystemInterface::EXISTS_RENAME);
      
      if (!$final_uri) {
        throw new \Exception('Failed to save file to destination');
      }

      // Create file entity
      $file_entity = File::create([
        'uid' => $this->currentUser->id(),
        'filename' => basename($final_uri),
        'uri' => $final_uri,
        'filesize' => $file_size,
        'filemime' => $file_mime,
        'status' => 1,
      ]);
      $file_entity->save();

      $logger->info('File entity created with ID: @fid and URI: @uri', [
        '@fid' => $file_entity->id(), 
        '@uri' => $file_entity->getFileUri()
      ]);
      
      // Mark file as in use
      $this->fileUsage->add($file_entity, 'dumi', 'file', $file_entity->id());
      
    } catch (\Exception $e) {
      $logger->error('File upload failed. Error: @message', ['@message' => $e->getMessage()]);
      return new JsonResponse(['error' => 'File upload failed. Please try again.'], 500);
    }

    // Create media entity
    try {
      $media_bundle = $request->request->get('media_bundle', 'image');
      $preview_style = $request->request->get('preview_style', '');
      
      // Determine which field to use for the image
      $image_field_names = ['field_media_image', 'field_image', 'image'];
      
      $media_storage = $this->entityTypeManager->getStorage('media');
      $sample_media = $media_storage->create(['bundle' => $media_bundle]);
      
      $media_field = null;
      foreach ($image_field_names as $field_name) {
        if ($sample_media->hasField($field_name)) {
          $media_field = $field_name;
          break;
        }
      }
      
      if (!$media_field) {
        $logger->error('No valid image field found on media bundle @bundle', ['@bundle' => $media_bundle]);
        // Clean up the uploaded file
        $file_entity->delete();
        return new JsonResponse(['error' => 'Media configuration error. Please contact administrator.'], 500);
      }

      // Create media entity
      $media = $this->entityTypeManager->getStorage('media')->create([
        'bundle' => $media_bundle,
        'uid' => $this->currentUser->id(),
        'status' => 1,
        'name' => $file_entity->getFilename() ?? 'Uploaded image',
        $media_field => [
          'target_id' => $file_entity->id(),
          'alt' => $file_entity->getFilename() ?? 'Uploaded image',
        ],
      ]);
      $media->save();

      $logger->info('Media entity created with ID: @mid and file reference: @fid', [
        '@mid' => $media->id(), 
        '@fid' => $file_entity->id()
      ]);
      
      // Update file usage to reference media entity
      $this->fileUsage->add($file_entity, 'dumi', 'media', $media->id());

      // Generate preview URL
      $preview_url = $this->fileUrlGenerator->generateString($final_uri);
      
      // Apply image style if specified
      if (!empty($preview_style) && \Drupal::moduleHandler()->moduleExists('image')) {
        try {
          $image_style = ImageStyle::load($preview_style);
          if ($image_style) {
            $preview_url = $image_style->buildUrl($final_uri);
          }
        } catch (\Exception $e) {
          $logger->warning('Failed to apply image style @style: @message', [
            '@style' => $preview_style,
            '@message' => $e->getMessage()
          ]);
          // Continue with original URL
        }
      }

      return new JsonResponse([
        'media_id' => $media->id(),
        'file_id' => $file_entity->id(),
        'filename' => $file_entity->getFilename(),
        'filesize' => $file_size,
        'preview_url' => $preview_url,
      ]);

    } catch (\Exception $e) {
      $logger->error('Media entity creation failed. Error: @message', ['@message' => $e->getMessage()]);
      
      // Clean up the uploaded file if media creation failed
      try {
        if (isset($file_entity) && $file_entity) {
          $file_entity->delete();
        }
      } catch (\Exception $cleanup_error) {
        $logger->error('Failed to clean up file after media creation error: @message', [
          '@message' => $cleanup_error->getMessage()
        ]);
      }
      
      return new JsonResponse(['error' => 'An error occurred while creating the media entity.'], 500);
    }
  }

}