/**
 * @file
 * JavaScript for the Dumi image upload widget with hidden slots, auto-expand,
 * and "Remove All" functionality.
 */

/**
 * Hide empty rows immediately on DOM load
 * This runs BEFORE Drupal.behaviors to prevent flash of visible rows
 */
(function() {
  'use strict';
  
  // Run as early as possible
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', hideEmptyRowsEarly);
  } else {
    hideEmptyRowsEarly();
  }
  
  function hideEmptyRowsEarly() {
    // Find all DUMI table rows
    const rows = document.querySelectorAll('.field-multiple-table tr.draggable');
    
    rows.forEach(function(row, index) {
      // Skip first row (delta 0)
      if (index === 0) return;
      
      // Check if row has a container with an image
      const container = row.querySelector('.dumi-upload-container');
      if (!container) return;
      
      const hasImage = container.classList.contains('has-image');
      const targetInput = row.querySelector('.dumi-media-id-input');
      const hasValue = targetInput && targetInput.value && targetInput.value !== '';
      
      // Only hide if truly empty
      if (!hasImage && !hasValue) {
        row.classList.add('dumi-hidden-row', 'dumi-row-force-hidden');
        
        // Also hide the container
        if (container && !container.classList.contains('dumi-hidden-slot')) {
          container.classList.add('dumi-hidden-slot');
        }
      } else {
        // Has value/image - ensure it's visible
        row.classList.remove('dumi-hidden-row', 'dumi-row-force-hidden');
        row.classList.add('dumi-row-in-use');
        container.classList.remove('dumi-hidden-slot');
        container.classList.add('dumi-slot-in-use');
      }
    });
  }
})();

(function ($, Drupal, once) {
  'use strict';

  /**
   * Helper: Ensure one empty row is visible for continued uploads
   */
  function ensureOneEmptyRowVisible($widget) {
    const fieldName = $widget.data('field-name');
    if (!fieldName) {
      const $firstContainer = $widget.find('.dumi-upload-container').first();
      fieldName = $firstContainer.data('field-name');
    }
    
    if (!fieldName) return;
    
    // Find all containers for this field
    const $allContainers = $widget.find('.dumi-upload-container[data-field-name="' + fieldName + '"]');
    
    // Check if we have ANY images at all
    const $containersWithImages = $allContainers.filter('.has-image');
    const hasAnyImages = $containersWithImages.length > 0;
    
    // If no images at all, don't show continuation row (back to initial state)
    if (!hasAnyImages) {
      console.log('[DUMI] No images present - hiding all non-first rows');
      // Hide all rows except first
      $allContainers.each(function() {
        const $container = $(this);
        if (!$container.hasClass('dumi-delta-first')) {
          forceHideRow($container);
        }
      });
      return;
    }
    
    // have images - find first empty container (no value, no image)
    let $firstEmptyContainer = null;
    
    $allContainers.each(function() {
      const $container = $(this);
      const $targetInput = $container.find('.dumi-media-id-input');
      const hasValue = $targetInput.val() && $targetInput.val() !== '';
      const hasImage = $container.hasClass('has-image');
      
      if (!hasValue && !hasImage && !$firstEmptyContainer) {
        $firstEmptyContainer = $container;
        return false; // break loop
      }
    });
    
    // If we found an empty container, make sure it's visible
    if ($firstEmptyContainer && !$firstEmptyContainer.hasClass('dumi-delta-first')) {
      console.log('[DUMI] Showing one empty row for continued uploads');
      forceShowRow($firstEmptyContainer);
    }
  }

  /**
   * Helper: Force show a row (remove all hiding)
   */
  function forceShowRow($container) {
    const $tableRow = $container.closest('tr');
    if ($tableRow.length) {
      // Remove ALL hiding classes
      $tableRow.removeClass('dumi-hidden-row dumi-row-force-hidden')
               .addClass('dumi-row-in-use')
               .css('display', ''); // Clear inline styles
      
      // Remove display:none if it was set
      $tableRow.show();
    }
    
    // Remove hiding from container
    $container.removeClass('dumi-hidden-slot')
              .addClass('dumi-slot-in-use');
  }

  /**
   * Helper: Force hide a row (only if not delta 0)
   */
  function forceHideRow($container) {
    const isDeltaFirst = $container.hasClass('dumi-delta-first');
    
    if (!isDeltaFirst) {
      const $tableRow = $container.closest('tr');
      if ($tableRow.length) {
        $tableRow.addClass('dumi-hidden-row dumi-row-force-hidden')
                 .removeClass('dumi-row-in-use')
                 .hide();
      }
      
      $container.addClass('dumi-hidden-slot')
                .removeClass('dumi-slot-in-use');
    }
  }

  /**
   * Helper: Get or create progress bar at widget level (above table)
   */
  function getProgressBar($container) {
    const $widget = $container.closest('.field--widget-dumi-widget');
    let $statusBar = $widget.find('> .dumi-multi-status').first();
    
    if (!$statusBar.length) {
      $statusBar = $widget.find('> div > .dumi-multi-status').first();
    }
    
    if (!$statusBar.length) {
      $statusBar = $(`
        <div class="dumi-multi-status hidden">
          <div class="dumi-progress-text">Preparing upload...</div>
          <div class="dumi-progress-bar">
            <div class="dumi-progress-fill"></div>
          </div>
        </div>
      `);
      $widget.prepend($statusBar);
    }
    
    return $statusBar;
  }

  /**
   * Show progress bar with initial message
   */
  function showProgressBar($container, message) {
    const $statusBar = getProgressBar($container);
    $statusBar.removeClass('hidden');
    if (message) {
      $statusBar.find('.dumi-progress-text').text(message);
    }
    return $statusBar;
  }

  /**
   * Hide progress bar
   */
  function hideProgressBar($container) {
    const $statusBar = getProgressBar($container);
    $statusBar.addClass('hidden');
  }

  /**
   * Get widget-level message container
   */
  function getWidgetMessages($container) {
    const $widget = $container.closest('.field--widget-dumi-widget');
    let $messages = $widget.find('> .dumi-widget-messages').first();
    
    if (!$messages.length) {
      $messages = $widget.find('> div > .dumi-widget-messages').first();
    }
    
    if (!$messages.length) {
      $messages = $('<div class="dumi-widget-messages hidden"></div>');
      const $progressBar = getProgressBar($container);
      
      if ($progressBar.length) {
        $progressBar.after($messages);
      } else {
        $widget.prepend($messages);
      }
    }
    
    return $messages;
  }

  /**
   * Show widget-level message
   */
  function showWidgetMessage($container, message, type) {
    const $messages = getWidgetMessages($container);
    $messages.removeClass('hidden error success')
      .addClass(type)
      .text(message);
  }

  /**
   * Hide widget-level messages
   */
  function hideWidgetMessages($container) {
    const $messages = getWidgetMessages($container);
    $messages.addClass('hidden').text('').removeClass('error success');
  }

  /**
   * Update Remove All button visibility
   * Only show if there are 2 or more images
   */
  function updateRemoveAllButtonVisibility($widget) {
    if (!$widget || !$widget.length) {
      return;
    }
    
    // Button is in dumi_actions_wrapper within the widget (non-numeric key in $elements)
    const $button = $widget.find('.dumi-remove-all-button');
    
    if (!$button.length) {
      return;
    }
    
    const $containersWithImages = $widget.find('.dumi-upload-container.has-image');
    
    // Show button only if there are 2 or more images
    if ($containersWithImages.length >= 2) {
      $button.show().prop('disabled', false).css('display', 'inline-block');
    } else {
      $button.hide().prop('disabled', true).css('display', 'none');
    }
  }

  /**
   * Updates the UI state
   */
  function updateUIState($container, state, defaultLabelContent) {
    const $previewContainer = $container.find('.dumi-upload-preview');
    const $labelContainer = $container.find('.dumi-upload-label-container label');
    const $removeButton = $container.find('.dumi-remove-button');

    switch(state) {
      case 'default':
        $container.removeClass('has-image');
        $labelContainer.html(defaultLabelContent).show();
        $previewContainer.empty();
        break;
        
      case 'uploading':
        $container.removeClass('has-image');
        $labelContainer.html(defaultLabelContent).show();
        // Show row during upload
        forceShowRow($container);
        break;
        
      case 'uploaded':
        $container.addClass('has-image');
        if ($removeButton.length) {
          $removeButton.css('display', 'block');
        }
        // Force show row when image is uploaded
        forceShowRow($container);
        break;
    }
  }

  /**
   * Check remove button icon
   */
  function checkRemoveButtonIcon($removeButton) {
    const img = new Image();
    img.onerror = function() {
      $removeButton.addClass('fallback-svg');
    };
    img.src = '/modules/custom/dumi/images/x_bkgd.png';
  }

  /**
   * Attach remove functionality
   */
  function attachRemoveFunctionality($removeButton, $container, defaultLabelContent) {
    $removeButton.off('click.dumi-remove');
    
    $removeButton.on('click.dumi-remove', function (e) {
      e.preventDefault();
      e.stopPropagation();
      e.stopImmediatePropagation();

      const $input = $container.find('.dumi-upload-file-input');
      const $targetIdInput = $container.find('.dumi-media-id-input');
      const $errorMessageContainer = $container.find('.dumi-error-message');
      const $widget = $container.closest('.field--widget-dumi-widget');

      $targetIdInput.val('');
      $input.val('');
      $errorMessageContainer.text('');
      $targetIdInput.trigger('change');

      updateUIState($container, 'default', defaultLabelContent);
      
      // Hide row using helper function
      forceHideRow($container);
      
      // Update button visibility after removal
      setTimeout(function() {
        updateRemoveAllButtonVisibility($widget);
        
        // Check if any images remain
        const stillHasImages = $widget.find('.dumi-upload-container.has-image').length > 0;
        
        if (stillHasImages) {
          // Still have images - show one empty row for continuation
          ensureOneEmptyRowVisible($widget);
        } else {
          // No images left - back to initial state, hide all non-first rows
          console.log('[DUMI] Last image removed - hiding all non-first rows');
          const fieldName = $widget.data('field-name') || $container.data('field-name');
          if (fieldName) {
            $widget.find('.dumi-upload-container[data-field-name="' + fieldName + '"]').each(function() {
              const $c = $(this);
              if (!$c.hasClass('dumi-delta-first')) {
                forceHideRow($c);
              }
            });
          }
        }
      }, 50);
      
      return false;
    });
  }

  /**
   * Helper functions for messages
   */
  function showError($container, message) {
    showWidgetMessage($container, message, 'error');
  }

  function showSuccess($container, message) {
    showWidgetMessage($container, message, 'success');
  }

  function clearMessages($container) {
    hideWidgetMessages($container);
  }

  /**
   * Main widget behavior
   */
  Drupal.behaviors.dumiWidget = {
    attach: function (context) {
      
      once('dumiTableRowInit', 'tr.dumi-hidden-row', context).forEach(function (row) {
        const $row = $(row);
        const $container = $row.find('.dumi-upload-container');
        const $targetInput = $container.find('.dumi-media-id-input');
        
        // If row has a value, ensure it's visible
        if ($targetInput.val() && $targetInput.val() !== '') {
          $row.removeClass('dumi-hidden-row dumi-row-force-hidden').addClass('dumi-row-in-use').show();
          $container.removeClass('dumi-hidden-slot').addClass('dumi-slot-in-use has-image');
        } else {
          // Truly empty - keep hidden
          $row.hide();
        }
      });
      
      once('dumiWidget', '.dumi-upload-file-input', context).forEach(function (element) {
        const $input = $(element);
        const $container = $input.closest('.dumi-upload-container');
        const $errorMessageContainer = $container.find('.dumi-error-message');
        const $targetIdInput = $container.find('.dumi-media-id-input');
        const $previewContainer = $container.find('.dumi-upload-preview');
        const $widget = $container.closest('.field--widget-dumi-widget');

        const fieldName = $container.data('field-name');
        const delta = parseInt($container.data('delta')) || 0;
        const cardinality = parseInt($container.data('cardinality')) || 1;
        const isMultiple = (cardinality > 1 || cardinality === -1);
        const defaultLabelContent = $container.data('default-label') || 'Click to upload image';
        const hideEmptySlots = $container.data('hide-empty-slots') !== false;
        const autoExpandSlots = $container.data('auto-expand-slots') !== false;

        let $removeButton = $container.find('.dumi-remove-button');
        if (!$removeButton.length) {
          $removeButton = $('<button type="button" class="dumi-remove-button" data-drupal-no-form-submit="true" title="Remove image" aria-label="Remove image"></button>');
          $previewContainer.append($removeButton);
          checkRemoveButtonIcon($removeButton);
          attachRemoveFunctionality($removeButton, $container, defaultLabelContent);
        } else {
          attachRemoveFunctionality($removeButton, $container, defaultLabelContent);
        }

        const hasImage = $targetIdInput.val() && $targetIdInput.val() !== '';
        updateUIState($container, hasImage ? 'uploaded' : 'default', defaultLabelContent);

        if (hasImage) {
          if ($removeButton.parent().length === 0 || !$removeButton.parent().hasClass('dumi-upload-preview')) {
            $previewContainer.append($removeButton);
          }
          $removeButton.css('display', 'block');
        }
        
        // Update button visibility on initial load
        updateRemoveAllButtonVisibility($widget);
        
        // Show one empty row on initial load (for continued uploads)
        if (isMultiple) {
          ensureOneEmptyRowVisible($widget);
        }

        let isClickHandlerActive = false;
        
        $container.on('click.dumi-label', '.dumi-upload-label, .dumi-change-text', function(e) {
          e.preventDefault();
          e.stopPropagation();
          
          if (isClickHandlerActive || $(e.target).closest('.dumi-remove-button').length) {
            return false;
          }
          
          isClickHandlerActive = true;
          setTimeout(function() {
            $input.trigger('click');
            setTimeout(function() {
              isClickHandlerActive = false;
            }, 100);
          }, 10);
          
          return false;
        });

        $container.on('click.dumi-preview', '.dumi-upload-preview', function(e) {
          e.preventDefault();
          e.stopPropagation();
          
          if (!$container.hasClass('has-image') || 
              $(e.target).closest('.dumi-remove-button').length || 
              isClickHandlerActive) {
            return false;
          }
          
          isClickHandlerActive = true;
          setTimeout(function() {
            $input.trigger('click');
            setTimeout(function() {
              isClickHandlerActive = false;
            }, 100);
          }, 10);
          
          return false;
        });

        let isUploading = false;

        $input.on('change.dumi-upload', function (e) {
          const files = this.files;
          
          if (isUploading || !files || files.length === 0 || !$container.length) {
            return;
          }

          if (isMultiple && files.length > 1) {
            if (hideEmptySlots && autoExpandSlots) {
              handleMultiFileUploadWithExpansion(files, $container, fieldName, cardinality);
            } else {
              handleMultiFileUpload(files, $container, fieldName, cardinality);
            }
          } else {
            const file = files[0];
            isUploading = true;
            handleSingleFileUpload(file, $container, delta, function() {
              isUploading = false;
              updateRemoveAllButtonVisibility($widget);
            });
          }

          $input.val('');
        });
      });

      once('dumiWidgetRemove', '.dumi-remove-button', context).forEach(function (element) {
        const $removeButton = $(element);
        const $container = $removeButton.closest('.dumi-upload-container');
        const defaultLabelContent = $container.data('default-label') || 'Click to upload image';
        attachRemoveFunctionality($removeButton, $container, defaultLabelContent);
      });
    }
  };

  /**
   * Remove All Button Behavior with proper field name detection
   */
  Drupal.behaviors.dumiRemoveAll = {
    attach: function (context) {
      once('dumiRemoveAll', '.dumi-remove-all-button', context).forEach(function (button) {
        const $button = $(button);
        const $widget = $button.closest('.field--widget-dumi-widget');
        
        if (!$widget.length) {
          console.error('DUMI Error - Cannot find widget for Remove All button');
          return;
        }
        
        //  Get field name from multiple sources
        let fieldName = $widget.data('field-name');
        if (!fieldName) {
          const $firstContainer = $widget.find('.dumi-upload-container').first();
          fieldName = $firstContainer.data('field-name');
        }
        if (!fieldName) {
          fieldName = $button.data('field-name');
        }
        
        if (!fieldName) {
          console.error('DUMI Error - Cannot find field name for Remove All button');
          return;
        }
        
        let removeAllInProgress = false;
        
        // Prevent double-binding
        $button.off('click.dumi-remove-all');
        
        $button.on('click.dumi-remove-all', function(e) {
          e.preventDefault();
          e.stopPropagation();
          e.stopImmediatePropagation();
          
          if (removeAllInProgress) {
            return false;
          }
          
          const $allContainers = $widget.find('.dumi-upload-container[data-field-name="' + fieldName + '"]');
          const $containersWithImages = $allContainers.filter('.has-image');
          
          if ($containersWithImages.length === 0) {
            const $firstContainer = $allContainers.first();
            if ($firstContainer.length) {
              showWidgetMessage($firstContainer, 'No images to remove.', 'error');
              setTimeout(() => hideWidgetMessages($firstContainer), 2000);
            }
            return false;
          }
          
          const confirmMessage = 'Are you sure you want to remove all ' + $containersWithImages.length + 
                                ' image(s)? This cannot be undone.';
          if (!confirm(confirmMessage)) {
            return false;
          }
          
          removeAllInProgress = true;
          $button.prop('disabled', true).addClass('is-processing');
          
          let removedCount = 0;
          let processedCount = 0;
          const totalToRemove = $containersWithImages.length;
          
          $containersWithImages.each(function(index) {
            const $container = $(this);
            const $removeBtn = $container.find('.dumi-remove-button');
            
            setTimeout(function() {
              if ($removeBtn.length) {
                $removeBtn.trigger('click');
                removedCount++;
              }
              
              processedCount++;
              
              if (processedCount === totalToRemove) {
                removeAllInProgress = false;
                $button.prop('disabled', false).removeClass('is-processing');
                
                // Update button visibility after all removals
                setTimeout(function() {
                  updateRemoveAllButtonVisibility($widget);
                  
                  const $firstContainer = $allContainers.first();
                  if (removedCount > 0 && $firstContainer.length) {
                    showWidgetMessage($firstContainer, 
                                    'Successfully removed ' + removedCount + ' image(s).', 
                                    'success');
                    setTimeout(() => hideWidgetMessages($firstContainer), 4000);
                  }
                  
                  // After Remove All, check if we should show continuation row
                  // This will hide all extra rows if no images remain
                  ensureOneEmptyRowVisible($widget);
                  
                  // Double-check: if truly no images, hide all non-first rows
                  const stillHasImages = $widget.find('.dumi-upload-container.has-image').length > 0;
                  if (!stillHasImages) {
                    console.log('[DUMI] Remove All complete - hiding all non-first rows');
                    $allContainers.each(function() {
                      const $container = $(this);
                      if (!$container.hasClass('dumi-delta-first')) {
                        forceHideRow($container);
                      }
                    });
                  }
                }, 200);
              }
            }, index * 100);
          });
          
          return false;
        });
      });
    }
  };

  /**
   * Re-initialize after AJAX updates
   */
  Drupal.behaviors.dumiAjaxReinit = {
    attach: function (context, settings) {
      // This runs after Drupal AJAX operations
      $(document).once('dumiAjaxInit').on('ajaxComplete', function(event, xhr, ajaxSettings) {
        // Wait for DOM to settle after AJAX
        setTimeout(function() {
          // Re-check all widgets for button visibility
          $('.field--widget-dumi-widget', context).each(function() {
            updateRemoveAllButtonVisibility($(this));
          });
        }, 100);
      });
    }
  };

  /**
   * Handle multi-file upload WITH automatic slot expansion
   */
  function handleMultiFileUploadWithExpansion(files, $container, fieldName, cardinality) {
    const $statusBar = showProgressBar($container, 'Preparing upload slots for ' + files.length + ' files...');
    $statusBar.find('.dumi-progress-fill').css('width', '5%');

    const availableDeltas = findAvailableDeltas(fieldName);
    const neededSlots = files.length - availableDeltas.length;

    const MAX_BATCH_SIZE = 50;
    if (files.length > MAX_BATCH_SIZE) {
      showError($container, 'Too many files selected (' + files.length + '). Please upload in batches of ' + MAX_BATCH_SIZE + ' or fewer.');
      hideProgressBar($container);
      return;
    }

    if (neededSlots > 0) {
      expandSlotsForField(fieldName, neededSlots, function(success) {
        if (success) {
          const $statusBar = getProgressBar($container);
          $statusBar.find('.dumi-progress-text').text('Starting upload of ' + files.length + ' files...');
          $statusBar.find('.dumi-progress-fill').css('width', '10%');
          
          setTimeout(function() {
            const updatedDeltas = findAvailableDeltas(fieldName);
            proceedWithMultiUpload(files, updatedDeltas, fieldName, $container);
          }, 500);
        } else {
          showError($container, 'Unable to create enough upload slots. Try uploading ' + availableDeltas.length + ' files at a time.');
          hideProgressBar($container);
        }
      });
    } else {
      proceedWithMultiUpload(files, availableDeltas, fieldName, $container);
    }
  }

  /**
   * Expand slots by triggering "Add more" button
   */
  function expandSlotsForField(fieldName, count, callback) {
    const buttonName = fieldName + '_add_more';
    const $addMoreButton = $('input[name$="_' + buttonName + '"]');

    if (!$addMoreButton.length) {
      console.error('DUMI Error - Add more button not found');
      callback(false);
      return;
    }

    triggerAddMoreSequentially($addMoreButton, count, 0, callback);
  }

  /**
   * Trigger "Add more" button sequentially
   */
  function triggerAddMoreSequentially($button, totalClicks, currentClick, callback) {
    if (currentClick >= totalClicks) {
      callback(true);
      return;
    }

    $button.trigger('mousedown').trigger('mouseup').trigger('click');

    const ajaxCompleteHandler = function(event, xhr, settings) {
      if (settings.extraData && settings.extraData._triggering_element_name === $button.attr('name')) {
        $(document).off('ajaxComplete', ajaxCompleteHandler);
        
        setTimeout(function() {
          triggerAddMoreSequentially($button, totalClicks, currentClick + 1, callback);
        }, 200);
      }
    };

    $(document).on('ajaxComplete', ajaxCompleteHandler);

    setTimeout(function() {
      $(document).off('ajaxComplete', ajaxCompleteHandler);
      triggerAddMoreSequentially($button, totalClicks, currentClick + 1, callback);
    }, 2000);
  }

  /**
   * Proceed with multi-file upload with proper visibility updates
   */
  function proceedWithMultiUpload(files, availableDeltas, fieldName, $container) {
    if (availableDeltas.length === 0) {
      showError($container, 'No empty slots available.');
      hideProgressBar($container);
      return;
    }

    const filesToUpload = Array.from(files).slice(0, availableDeltas.length);
    
    if (files.length > availableDeltas.length) {
      showError($container, 'Only ' + availableDeltas.length + ' file(s) can be uploaded. ' + 
                (files.length - availableDeltas.length) + ' will be skipped.');
    }

    // Force show all rows that will receive images BEFORE uploading
    for (let i = 0; i < filesToUpload.length; i++) {
      const $containerToShow = availableDeltas[i].$container;
      forceShowRow($containerToShow);
    }

    uploadFilesSequentially(filesToUpload, availableDeltas, fieldName, $container, function(results) {
      hideProgressBar($container);

      const successCount = results.filter(r => r.success).length;
      const failCount = results.filter(r => !r.success).length;
      
      let message = successCount + ' file(s) uploaded successfully.';
      if (failCount > 0) {
        message += ' ' + failCount + ' failed.';
      }
      
      showSuccess($container, message);
      setTimeout(() => clearMessages($container), 5000);
      
      const $widget = $container.closest('.field--widget-dumi-widget');
      
      // Update button visibility and re-check row visibility
      setTimeout(function() {
        // Double-check that all containers with images have visible rows
        $widget.find('.dumi-upload-container.has-image').each(function() {
          forceShowRow($(this));
        });
        
        // Show one empty row for continued uploads
        ensureOneEmptyRowVisible($widget);
        
        updateRemoveAllButtonVisibility($widget);
        
        // Re-attach behaviors to ensure button works
        Drupal.attachBehaviors($widget[0]);
      }, 200);
    });
  }

  /**
   * Handle multiple file uploads (legacy mode)
   */
  function handleMultiFileUpload(files, $container, fieldName, cardinality) {
    showProgressBar($container, 'Preparing to upload ' + files.length + ' files...');

    const availableDeltas = findAvailableDeltas(fieldName);

    if (availableDeltas.length === 0) {
      showError($container, 'No empty slots available.');
      hideProgressBar($container);
      return;
    }

    const filesToUpload = Array.from(files).slice(0, availableDeltas.length);
    
    if (files.length > availableDeltas.length) {
      showError($container, 'Only ' + availableDeltas.length + ' file(s) can be uploaded.');
    }

    uploadFilesSequentially(filesToUpload, availableDeltas, fieldName, $container, function(results) {
      hideProgressBar($container);

      const successCount = results.filter(r => r.success).length;
      const failCount = results.filter(r => !r.success).length;
      
      let message = successCount + ' file(s) uploaded.';
      if (failCount > 0) {
        message += ' ' + failCount + ' failed.';
      }
      
      showSuccess($container, message);
      setTimeout(() => clearMessages($container), 5000);
      
      const $widget = $container.closest('.field--widget-dumi-widget');
      
      // Update button visibility and show one empty row
      setTimeout(function() {
        updateRemoveAllButtonVisibility($widget);
        ensureOneEmptyRowVisible($widget);
      }, 200);
    });
  }

  /**
   * Find all empty delta slots
   */
  function findAvailableDeltas(fieldName) {
    const availableDeltas = [];

    $('.dumi-upload-container[data-field-name="' + fieldName + '"]').each(function() {
      const $container = $(this);
      const containerDelta = parseInt($container.data('delta'));
      const $targetInput = $container.find('.dumi-media-id-input');
      const currentValue = $targetInput.val();

      if (!currentValue || currentValue === '') {
        availableDeltas.push({
          delta: containerDelta,
          $container: $container,
          $targetInput: $targetInput
        });
      }
    });

    availableDeltas.sort((a, b) => a.delta - b.delta);
    return availableDeltas;
  }

  /**
   * Upload files sequentially
   */
  function uploadFilesSequentially(files, deltaSlots, fieldName, $statusContainer, callback) {
    const results = [];
    let currentIndex = 0;
    const totalFiles = files.length;

    function uploadNext() {
      if (currentIndex >= files.length) {
        callback(results);
        return;
      }

      const file = files[currentIndex];
      const deltaSlot = deltaSlots[currentIndex];
      const fileNumber = currentIndex + 1;

      updateProgressUI($statusContainer, fileNumber, totalFiles, file.name);

      handleSingleFileUpload(file, deltaSlot.$container, deltaSlot.delta, function(success, mediaId) {
        results.push({
          file: file.name,
          delta: deltaSlot.delta,
          success: success,
          mediaId: mediaId
        });

        currentIndex++;
        
        // Small delay between uploads
        setTimeout(uploadNext, 100);
      });
    }

    uploadNext();
  }

  /**
   * Update progress UI
   */
  function updateProgressUI($container, current, total, fileName) {
    const $statusBar = getProgressBar($container);
    const percentage = Math.round((current / total) * 100);
    const shortName = fileName.length > 30 ? fileName.substring(0, 27) + '...' : fileName;

    $statusBar.find('.dumi-progress-text')
      .text('Uploading ' + current + ' of ' + total + ': ' + shortName);
    $statusBar.find('.dumi-progress-fill').css('width', percentage + '%');
  }

  /**
   * Single file upload
   */
  function handleSingleFileUpload(file, $container, delta, callback) {
    const $input = $container.find('.dumi-upload-file-input');
    const $targetIdInput = $container.find('.dumi-media-id-input');
    const $previewContainer = $container.find('.dumi-upload-preview');
    const $errorMessageContainer = $container.find('.dumi-error-message');
    const defaultLabelContent = $container.data('default-label');
    
    let $removeButton = $container.find('.dumi-remove-button');
    if (!$removeButton.length) {
      $removeButton = $('<button type="button" class="dumi-remove-button" data-drupal-no-form-submit="true" title="Remove image" aria-label="Remove image"></button>');
      $previewContainer.append($removeButton);
      checkRemoveButtonIcon($removeButton);
      attachRemoveFunctionality($removeButton, $container, defaultLabelContent);
    }

    const uploadUrl = $container.data('upload-url');
    const csrfToken = $container.data('csrf-token');
    const mediaBundle = $container.data('media-bundle');
    const previewStyle = $container.data('preview-image-style');

    const reader = new FileReader();
    reader.onload = function (e) {
      const previewHtml = '<img src="' + e.target.result + '" alt="Image preview" />';
      $previewContainer.html(previewHtml);
      
      if ($removeButton.length) {
        $previewContainer.append($removeButton);
        $removeButton.css('display', 'block');
        attachRemoveFunctionality($removeButton, $container, defaultLabelContent);
      }
      
      updateUIState($container, 'uploading', defaultLabelContent);
    };
    reader.readAsDataURL(file);

    $errorMessageContainer.text('');

    const formData = new FormData();
    formData.append('dumi_upload', file);
    formData.append('media_bundle', mediaBundle);
    formData.append('preview_style', previewStyle);
	
	const $altInput = $container.find('.dumi-alt-text-input');
    if ($altInput.length && $altInput.val()) {
    formData.append('alt_text', $altInput.val());
     } else {
    // Use filename as fallback
    formData.append('alt_text', file.name);
    }
  

    $container.addClass('is-loading');

    fetch(uploadUrl, {
        method: 'POST',
        body: formData,
        headers: {
          'X-CSRF-Token': csrfToken
        }
      })
      .then(response => {
        $container.removeClass('is-loading');
        if (!response.ok) {
          throw new Error('Network response was not ok');
        }
        return response.json();
      })
      .then(response => {
        if (response.media_id) {
          const mediaId = parseInt(response.media_id, 10);
          $targetIdInput.val(mediaId);
          $targetIdInput.trigger('change');
          
          if (response.preview_url) {
            const previewHtml = '<img src="' + response.preview_url + '" alt="Image preview" />';
            $previewContainer.html(previewHtml);
            
            if ($removeButton.length) {
              $previewContainer.append($removeButton);
              $removeButton.css('display', 'block');
              attachRemoveFunctionality($removeButton, $container, defaultLabelContent);
            }
          }
          
          updateUIState($container, 'uploaded', defaultLabelContent);
          
          // Update button visibility after upload
          const $widget = $container.closest('.field--widget-dumi-widget');
          setTimeout(function() {
            updateRemoveAllButtonVisibility($widget);
            // Show one empty row for continued uploads
            ensureOneEmptyRowVisible($widget);
          }, 50);
          
          if (callback) callback(true, mediaId);
        } else if (response.error) {
          $errorMessageContainer.text(response.error);
          updateUIState($container, 'default', defaultLabelContent);
          
          if (callback) callback(false, null);
        } else {
          $errorMessageContainer.text('An unexpected error occurred.');
          updateUIState($container, 'default', defaultLabelContent);
          
          if (callback) callback(false, null);
        }
      })
      .catch(error => {
        $container.removeClass('is-loading');
        $errorMessageContainer.text('A network error occurred. Please try again.');
        updateUIState($container, 'default', defaultLabelContent);
        
        if (callback) callback(false, null);
      });
  }
  /**
   * Delegate "Add another item" clicks from visible button to AJAX button
   */
  Drupal.behaviors.dumiAddMoreDelegate = {
    attach: function (context) {
      once('dumiAddMoreDelegate', '.dumi-buttons-wrapper .field-add-more-submit:not(.dumi-ajax-add-more)', context).forEach(function (button) {
        const $button = $(button);
      
        $button.on('click', function(e) {
          e.preventDefault();
          e.stopPropagation();
        
          // Find the hidden AJAX button
          const $widget = $button.closest('.field--widget-dumi-widget');
          const $ajaxButton = $widget.find('.dumi-ajax-add-more');
        
          if ($ajaxButton.length) {
            // Trigger the AJAX button
            $ajaxButton.trigger('mousedown').trigger('click');
          }
        
          return false;
        });
      });
    }
  };
})(jQuery, Drupal, once);