<?php

namespace Drupal\dumi\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;

/**
 * Configure Dumi settings.
 */
class DumiSettingsForm extends ConfigFormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The file URL generator.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected $fileUrlGenerator;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Constructs a new DumiSettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\File\FileUrlGeneratorInterface $file_url_generator
   *   The file URL generator.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config
   *   The typed configuration manager.
   */
  public function __construct(ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager, FileUrlGeneratorInterface $file_url_generator, LoggerChannelFactoryInterface $logger_factory, TypedConfigManagerInterface $typed_config) {
    parent::__construct($config_factory, $typed_config);
    $this->entityTypeManager = $entity_type_manager;
    $this->fileUrlGenerator = $file_url_generator;
    $this->loggerFactory = $logger_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('file_url_generator'),
      $container->get('logger.factory'),
      $container->get('config.typed')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dumi_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['dumi.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('dumi.settings');

    $form['file_size_limit'] = [
      '#type' => 'number',
      '#title' => $this->t('File Size Limit (in MB)'),
      '#default_value' => $config->get('file_size_limit') ?: 5,
      '#min' => 1,
      '#required' => TRUE,
    ];

    // Build the table of fields that are using the Dumi widget.
    $form['fields_table'] = [
      '#type' => 'details',
      '#title' => $this->t('Fields using the Dumi widget'),
      '#open' => TRUE,
    ];

    $dumi_fields = $this->findDumiFields();
    
    $header = [
      'entity_type' => $this->t('Entity Type'),
      'bundle' => $this->t('Content Type / Bundle'),
      'field_name' => $this->t('Field Name'),
      'form_mode' => $this->t('Form Mode'),
      'label_image' => $this->t('Custom Label Image'),
    ];

    $rows = [];
    foreach ($dumi_fields as $field_info) {
      $bundle = $field_info['bundle'];
      $custom_label_fid = $config->get('labels.' . $bundle);
      $custom_label_image = $this->t('No image');
      
      try {
        if ($custom_label_fid && $file = $this->entityTypeManager->getStorage('file')->load($custom_label_fid)) {
          $file_uri = $file->getFileUri();
          $style = 'style="max-width: 100px; height: auto;"';
          $custom_label_image = '<img src="' . $this->fileUrlGenerator->generate($file_uri)->toString() . '" alt="' . $file->getFilename() . '" ' . $style . ' />';
        }
      }
      catch (\Exception $e) {
        $custom_label_image = $this->t('Error loading image');
      }

      $rows[] = [
        'entity_type' => $field_info['entity_type'],
        'bundle' => $bundle,
        'field_name' => $field_info['field_name'],
        'form_mode' => $field_info['form_mode'],
        'label_image' => [
          'data' => [
            '#markup' => $custom_label_image,
          ],
        ],
      ];
    }
    
    if (empty($rows)) {
      $form['fields_table']['no_fields'] = [
        '#markup' => $this->t('No fields are currently configured to use the Dumi widget.'),
      ];
    } else {
      $form['fields_table']['table'] = [
        '#type' => 'table',
        '#header' => $header,
        '#rows' => $rows,
        '#empty' => $this->t('No fields found.'),
      ];
    }

    $form['custom_labels'] = [
      '#type' => 'details',
      '#title' => $this->t('Custom Widget Labels'),
      '#open' => TRUE,
    ];
    $this->addCustomLabelSection($form['custom_labels'], $dumi_fields);

    return parent::buildForm($form, $form_state);
  }

  /**
   * Adds the custom label upload section to the form.
   *
   * @param array $form
   *   The form array.
   * @param array $dumi_fields
   *   An array of fields using the Dumi widget.
   */
  protected function addCustomLabelSection(array &$form, array $dumi_fields) {
    $config = $this->config('dumi.settings');
    $bundles = array_unique(array_column($dumi_fields, 'bundle'));

    if (empty($bundles)) {
      $form['info'] = [
        '#markup' => $this->t('No Dumi fields are currently in use on any content type.'),
      ];
      return;
    }

    $form['info'] = [
      '#markup' => $this->t('Upload a custom label image for each content type. Leave blank to use the default text.'),
    ];

    foreach ($bundles as $bundle) {
      // Handle different entity types, not just 'node_type'
      $bundle_info = NULL;
      $bundle_label = $bundle;
      
      // Try to load bundle info for different entity types
      try {
        // Check if it's a node bundle
        if ($this->entityTypeManager->hasDefinition('node_type')) {
          $bundle_info = $this->entityTypeManager->getStorage('node_type')->load($bundle);
        }
        
        // If not found, try other common entity types
        if (!$bundle_info) {
          $entity_type_bundles = [
            'taxonomy_vocabulary',
            'media_type', 
            'block_content_type',
            'commerce_product_type'
          ];
          
          foreach ($entity_type_bundles as $bundle_type) {
            if ($this->entityTypeManager->hasDefinition($bundle_type)) {
              $bundle_info = $this->entityTypeManager->getStorage($bundle_type)->load($bundle);
              if ($bundle_info) {
                break;
              }
            }
          }
        }
        
        if ($bundle_info) {
          $bundle_label = $bundle_info->label();
        }
      }
      catch (\Exception $e) {
        $this->loggerFactory->get('dumi')->warning('Could not load bundle info for @bundle: @message', [
          '@bundle' => $bundle,
          '@message' => $e->getMessage(),
        ]);
      }

      $fid = $config->get('labels.' . $bundle);
      $max_size_mb = (int) $config->get('file_size_limit') ?: 5;
      $max_size_bytes = $max_size_mb * 1024 * 1024;

      $form[$bundle] = [
        '#type' => 'managed_file',
        '#title' => $this->t('Label for %bundle', ['%bundle' => $bundle_label]),
        '#upload_location' => 'public://dumi_labels',
        '#default_value' => $fid ? [$fid] : [],
        '#upload_validators' => [
          'FileExtension' => [
            'extensions' => 'jpg jpeg png gif',
          ],
          'FileSizeLimit' => [
            'fileLimit' => $max_size_bytes,
          ],
        ],
      ];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);
    
    // Add validation for file size limit
    $file_size_limit = $form_state->getValue('file_size_limit');
    if ($file_size_limit < 1 || $file_size_limit > 100) {
      $form_state->setErrorByName('file_size_limit', $this->t('File size limit must be between 1 and 100 MB.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('dumi.settings');
    $config->set('file_size_limit', $form_state->getValue('file_size_limit'));
    
    $bundles = $this->findDumiFields();
    $bundles = array_unique(array_column($bundles, 'bundle'));
    
    foreach ($bundles as $bundle) {
      $fid_array = $form_state->getValue($bundle);
      $fid = NULL;
      
      // Handle both array and single value formats
      if (is_array($fid_array) && !empty($fid_array)) {
        $fid = reset($fid_array); // Get first element
      } elseif (!is_array($fid_array) && !empty($fid_array)) {
        $fid = $fid_array;
      }
      
      $config->set('labels.' . $bundle, $fid);
      
      if ($fid) {
        try {
          $file = $this->entityTypeManager->getStorage('file')->load($fid);
          if ($file) {
            // Make the file permanent and add usage
            $file->setPermanent();
            $file->save();
            \Drupal::service('file.usage')->add($file, 'dumi', 'dumi_settings', $this->getFormId());
          }
        }
        catch (\Exception $e) {
          $this->loggerFactory->get('dumi')->error('Failed to add file usage for FID @fid: @message', [
            '@fid' => $fid,
            '@message' => $e->getMessage(),
          ]);
        }
      }
    }
    
    $config->save();
    
    // Clear cache to ensure config changes take effect
    \Drupal::service('cache.config')->deleteAll();
    
    parent::submitForm($form, $form_state);
  }

  /**
   * Finds all fields that are using the Dumi widget.
   *
   * @return array
   *   An array of field information.
   */
  protected function findDumiFields() {
    $dumi_fields = [];
    
    try {
      $form_displays = $this->entityTypeManager->getStorage('entity_form_display')->loadMultiple();

      foreach ($form_displays as $display) {
        $components = $display->getComponents();
        foreach ($components as $field_name => $component) {
          if (isset($component['type']) && $component['type'] === 'dumi_widget') {
            $field_config_id = $display->getTargetEntityTypeId() . '.' . $display->getTargetBundle() . '.' . $field_name;
            
            // Check if field config exists before trying to load it
            if ($this->entityTypeManager->getStorage('field_config')->load($field_config_id)) {
              $dumi_fields[] = [
                'entity_type' => $display->getTargetEntityTypeId(),
                'bundle' => $display->getTargetBundle(),
                'field_name' => $field_name,
                'form_mode' => $display->getMode(),
              ];
            }
          }
        }
      }
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('dumi')->error('Error finding Dumi fields: @message', [
        '@message' => $e->getMessage(),
      ]);
    }

    return $dumi_fields;
  }
}