<?php

namespace Drupal\dumi\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Url;
use Drupal\Core\Link;
use Drupal\Core\Render\Markup;
use Drupal\media\Entity\Media;
use Drupal\file\Entity\File;
use Drupal\image\Entity\ImageStyle;
use Drupal\Core\Access\CsrfTokenGenerator;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Component\Utility\Html;
use Drupal\Component\Utility\NestedArray;

/**
 * Plugin implementation of the 'dumi_widget' widget.
 *
 * @FieldWidget(
 *   id = "dumi_widget",
 *   label = @Translation("Direct Upload Media Image"),
 *   field_types = {
 *     "entity_reference"
 *   }
 * )
 */
class DumiWidget extends WidgetBase implements ContainerFactoryPluginInterface {

  protected $messenger;
  protected $csrfTokenGenerator;
  protected $fileUrlGenerator;
  protected $loggerFactory;
  protected $configFactory;
  protected $entityTypeManager;
  protected $currentUser;

  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, array $third_party_settings, MessengerInterface $messenger, CsrfTokenGenerator $csrf_token_generator, FileUrlGeneratorInterface $file_url_generator, LoggerChannelFactoryInterface $logger_factory, ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager, AccountProxyInterface $current_user) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
    $this->messenger = $messenger;
    $this->csrfTokenGenerator = $csrf_token_generator;
    $this->fileUrlGenerator = $file_url_generator;
    $this->loggerFactory = $logger_factory;
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->currentUser = $current_user;
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('messenger'),
      $container->get('csrf_token'),
      $container->get('file_url_generator'),
      $container->get('logger.factory'),
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('current_user')
    );
  }

  public static function defaultSettings() {
    return [
      'preview_image_style' => '',
      'file_size_limit_mb' => 0,
      'empty_slots_for_unlimited' => 25,
      'auto_expand_slots' => TRUE,
      'hide_empty_slots' => TRUE,
	  'show_alt_text_field' => TRUE,  
      'require_alt_text' => FALSE,    
    ] + parent::defaultSettings();
  }

  public function settingsForm(array $form, FormStateInterface $form_state) {
    $element = [];
	
	   // DUMI Widget Config Settings
       $element['configuration_info'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Global Configuration'),
        '#weight' => -100,
      ];
  
      $config_url = Url::fromRoute('dumi.settings');
      $config_link = Link::fromTextAndUrl(
        $this->t('Dumi Settings page'),
        $config_url
      );
  
      $element['configuration_info']['message'] = [
        '#type' => 'item',
        '#markup' => $this->t('<strong>Note:</strong> Customize widget form display with an image/icon and more: @link.', [
          '@link' => $config_link->toString(),
        ]),
        '#allowed_tags' => ['strong', 'a', 'em', 'br'],
      ];

    $element['preview_image_style'] = [
      '#type' => 'select',
      '#title' => $this->t('Form image style'),
      '#options' => image_style_options(TRUE),
      '#default_value' => $this->getSetting('preview_image_style'),
      '#description' => $this->t('Style used for the preview shown inside the widget. Leave empty for the original image.'),
    ];

    $element['file_size_limit_mb'] = [
      '#type' => 'number',
      '#title' => $this->t('File size limit (MB)'),
      '#default_value' => $this->getSetting('file_size_limit_mb'),
      '#description' => $this->t('Maximum file size in MB. Leave empty or 0 to use the global setting.'),
      '#min' => 0,
      '#step' => 1,
    ];

    $element['empty_slots_for_unlimited'] = [
      '#type' => 'number',
      '#title' => $this->t('Empty slots for unlimited fields'),
      '#default_value' => $this->getSetting('empty_slots_for_unlimited') ?: 10,
      '#description' => $this->t('Number of empty upload slots to display for unlimited cardinality fields. Default: 10'),
      '#min' => 1,
      '#max' => 100,
      '#step' => 1,
    ];

    $element['hide_empty_slots'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Hide empty slots until needed'),
      '#default_value' => $this->getSetting('hide_empty_slots') ?? TRUE,
      '#description' => $this->t('Only show the first upload slot. Additional slots will be created automatically when multiple files are selected.'),
    ];

    $element['auto_expand_slots'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto-expand slots for multi-file uploads'),
      '#default_value' => $this->getSetting('auto_expand_slots') ?? TRUE,
      '#description' => $this->t('Automatically create additional upload slots when multiple files are selected. Requires "Hide empty slots" to be enabled.'),
      '#states' => [
        'visible' => [
          ':input[name*="hide_empty_slots"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return $element;
  }

   /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    
    // settingsSummary() can ONLY show plain text - no HTML links allowed
    // Add plain text reference to settings page
    $summary[] = $this->t('Module settings: /admin/config/media/dumi');
    
    $style = $this->getSetting('preview_image_style');
    $summary[] = $style ? $this->t('Form image style: @s', ['@s' => $style]) : $this->t('Form image style: None');

    $size_limit = $this->getSetting('file_size_limit_mb');
    if ($size_limit > 0) {
      $summary[] = $this->t('File size limit: @size MB', ['@size' => $size_limit]);
    } else {
      $summary[] = $this->t('File size limit: Global setting');
    }

    $empty_slots = $this->getSetting('empty_slots_for_unlimited') ?: 10;
    $summary[] = $this->t('Empty slots for unlimited: @count', ['@count' => $empty_slots]);

    if ($this->getSetting('hide_empty_slots')) {
      $summary[] = $this->t('Empty slots: Hidden until needed');
    }

    if ($this->getSetting('auto_expand_slots')) {
      $summary[] = $this->t('Auto-expand: Enabled');
    }

    return $summary;
  }

  /**
   * Value callback to ensure target_id is properly formatted.
   */
  public function validateTargetId($element, $input, FormStateInterface $form_state) {
    if ($input === FALSE) {
      return $element['#default_value'];
    }
    return !empty($input) ? (int) $input : NULL;
  }

  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    if (!$this->currentUser->hasPermission('upload dumi files')) {
      $element['#markup'] = $this->t('You do not have permission to use this widget.');
      return $element;
    }

    $element['#attributes']['class'][] = 'dumi-widget-element';

    // Check field cardinality
    $field_storage = $this->fieldDefinition->getFieldStorageDefinition();
    $cardinality = $field_storage->getCardinality();
    $is_multiple = ($cardinality > 1 || $cardinality === -1);
    $is_first_delta = ($delta === 0);

    // Get existing media ID for this delta
    $field_item = $items->get($delta);
    $media_id = NULL;

    if ($field_item) {
      if (isset($field_item->target_id) && !empty($field_item->target_id)) {
        $media_id = (int) $field_item->target_id;
      } elseif (!$media_id) {
        $value = $field_item->getValue();
        $target_id = $value['target_id'] ?? NULL;
        if (!empty($target_id)) {
          $media_id = (int) $target_id;
        }
      }
    }

    // Load file URI and URL if media exists
    $file_uri = NULL;
    $file_url = NULL;

    if (!empty($media_id)) {
      try {
        $media = $this->entityTypeManager->getStorage('media')->load($media_id);
        if ($media && $media->access('view')) {
          $image_field_names = ['field_media_image', 'field_image', 'image'];
          foreach ($image_field_names as $field_name) {
            if ($media->hasField($field_name) && !$media->get($field_name)->isEmpty()) {
              $file = $media->get($field_name)->entity;
              if ($file instanceof File) {
                $file_uri = $file->getFileUri();
                $file_url = $this->fileUrlGenerator->generateString($file_uri);
                break;
              }
            }
          }
        }
      } catch (\Exception $e) {
        $this->loggerFactory->get('dumi')->error('Error loading media: @message', ['@message' => $e->getMessage()]);
        $media_id = NULL;
      }
    }

    // Get configuration
    $config = $this->configFactory->get('dumi.settings');
    $global_limit_mb = (int) $config->get('file_size_limit') ?: 5;
    $widget_file_size_limit = (int) $this->getSetting('file_size_limit_mb');
    $file_size_limit = $widget_file_size_limit > 0 ? $widget_file_size_limit : $global_limit_mb;
    $preview_style = $this->getSetting('preview_image_style') ?: '';
    $hide_empty_slots = $this->getSetting('hide_empty_slots') ?? TRUE;
    $auto_expand_slots = $this->getSetting('auto_expand_slots') ?? TRUE;

    $upload_url = Url::fromRoute('dumi.upload_file')->toString();
    $csrf_token = $this->csrfTokenGenerator->get('dumi_upload');
    $target_bundles = $this->getFieldSetting('handler_settings')['target_bundles'] ?? [];
    $media_bundle = !empty($target_bundles) ? reset($target_bundles) : 'image';

    $bundle = $this->getBundleFromContext($form, $form_state);
    $default_label_content = $this->getCustomLabelContent($bundle, $config, NULL, NULL);

    // Determine if this slot should be hidden
    $should_hide = FALSE;
    if ($hide_empty_slots && $is_multiple && !$is_first_delta && empty($media_id)) {
      $should_hide = TRUE;
    }

    // Container attributes
    $container_classes = [
      'dumi-upload-container',
      'js-form-wrapper',
      'form-wrapper',
      $media_id ? 'has-image' : '',
      $is_multiple ? 'dumi-multi-enabled' : 'dumi-single-only',
      $is_first_delta ? 'dumi-delta-first' : 'dumi-delta-' . $delta,
    ];

    if ($should_hide) {
      $container_classes[] = 'dumi-hidden-slot';
    }

    $element['upload_container'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => $container_classes,
        'data-upload-url' => $upload_url,
        'data-csrf-token' => $csrf_token,
        'data-file-size-limit' => $file_size_limit,
        'data-preview-image-style' => $preview_style,
        'data-media-bundle' => $media_bundle,
        'data-default-label' => $default_label_content,
        'data-field-name' => $this->fieldDefinition->getName(),
        'data-delta' => $delta,
        'data-cardinality' => $cardinality,
        'data-is-multiple' => $is_multiple ? 'true' : 'false',
        'data-hide-empty-slots' => $hide_empty_slots ? 'true' : 'false',
        'data-auto-expand-slots' => $auto_expand_slots ? 'true' : 'false',
      ],
    ];

    $element['upload_container']['target_id'] = [
      '#type' => 'hidden',
      '#default_value' => $media_id,
      '#attributes' => [
        'class' => ['dumi-media-id-input', 'dumi-delta-' . $delta . '-target-id'],
      ],
      '#value_callback' => [$this, 'validateTargetId'],
    ];

    // File input gets 'multiple' attribute for ALL deltas if field supports multiple
    $element['upload_container']['file'] = [
      '#type' => 'file',
      '#title' => $this->t('Choose a file'),
      '#title_display' => 'invisible',
      '#attributes' => [
        'class' => ['dumi-upload-file-input'],
        'id' => 'dumi-upload-file-' . $delta,
        'multiple' => $is_multiple, // Enable for all deltas, not just first
      ],
    ];

    $element['upload_container']['label_container'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['dumi-upload-label-container', 'js-form-wrapper', 'form-wrapper']
      ],
    ];

    // Generate preview markup
    $preview_markup = '';
    $label_content = '';
    
    if ($file_uri && $file_url) {
      if (!empty($preview_style) && \Drupal::moduleHandler()->moduleExists('image')) {
        try {
          $image_style = ImageStyle::load($preview_style);
          if ($image_style) {
            $styled_url = $image_style->buildUrl($file_uri);
            $preview_markup = '<img src="' . $styled_url . '" alt="' . $this->t('Image preview') . '" />';
          } else {
            $preview_markup = '<img src="' . $file_url . '" alt="' . $this->t('Image preview') . '" />';
          }
        } catch (\Exception $e) {
          $preview_markup = '<img src="' . $file_url . '" alt="' . $this->t('Image preview') . '" />';
        }
      } else {
        $preview_markup = '<img src="' . $file_url . '" alt="' . $this->t('Image preview') . '" />';
      }
      $label_content = $default_label_content;
    } else {
      if ($is_multiple && $is_first_delta) {
        $label_content = '<span class="dumi-multi-label">' . $this->t('Click to upload image(s)') . '</span>';
      } else {
        $label_content = $default_label_content;
      }
    }

    $element['upload_container']['label_container']['label'] = [
      '#type' => 'html_tag',
      '#tag' => 'label',
      '#value' => is_string($label_content) ? $label_content : '',
      '#markup' => is_string($label_content) ? $label_content : '',
      '#attributes' => [
        'class' => ['dumi-upload-label'],
        'role' => 'button',
        'tabindex' => '0',
        'aria-controls' => 'dumi-upload-file-' . $delta,
      ],
    ];

    $element['upload_container']['preview'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['dumi-upload-preview']],
      '#markup' => $preview_markup,
    ];

    $module_path = \Drupal::service('extension.list.module')->getPath('dumi');
    $remove_icon_url = '/' . $module_path . '/images/x_bkgd.png';
    
    $element['upload_container']['preview']['remove'] = [
      '#type' => 'button',
      '#value' => '',
      '#attributes' => [
        'class' => ['dumi-remove-button'],
        'data-drupal-no-form-submit' => 'true',
        'title' => $this->t('Remove image'),
        'aria-label' => $this->t('Remove image'),
        'style' => 'background-image: url(' . $remove_icon_url . ');',
      ],
    ];

    $element['upload_container']['error_message'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['dumi-error-message']],
    ];
	
// Add alt text field 
$show_alt_field = $this->getSetting('show_alt_text_field') ?? TRUE;
$require_alt = $this->getSetting('require_alt_text') ?? FALSE;

 if ($show_alt_field) {
   // Get existing alt text from media if it exists
   $existing_alt = '';
   if (!empty($media_id)) {
     try {
       $media = $this->entityTypeManager->getStorage('media')->load($media_id);
       if ($media && $media->access('view')) {
         $image_field_names = ['field_media_image', 'field_image', 'image'];
         foreach ($image_field_names as $field_name) {
           if ($media->hasField($field_name) && !$media->get($field_name)->isEmpty()) {
             $existing_alt = $media->get($field_name)->alt ?? '';
             break;
           }
         }
       }
     } catch (\Exception $e) {
       // Continue without alt text
      }
   }

   // Wrapper for collapsible alt text
   $element['upload_container']['alt_wrapper'] = [
     '#type' => 'container',
     '#attributes' => [
      'class' => ['dumi-alt-wrapper'],
     ],
     '#weight' => 5,
   ];

  // Use details element for native collapsible behavior
  $element['upload_container']['alt_wrapper']['details'] = [
    '#type' => 'details',
    '#title' => $this->t('Alternative Text'),
    '#open' => !empty($existing_alt), // Open if alt text exists
    '#attributes' => [
      'class' => ['dumi-alt-details'],
    ],
  ];

  $element['upload_container']['alt_wrapper']['details']['alt_text'] = [
    '#type' => 'textfield',
    '#title' => $this->t('Alt text'),
    '#title_display' => 'invisible',
    '#description' => $this->t('Describe this image for screen readers.'),
    '#default_value' => $existing_alt,
    '#maxlength' => 512,
    '#attributes' => [
      'class' => ['dumi-alt-text-input'],
      'placeholder' => $this->t('Describe this image'),
    ],
  ];

   // Make required if setting is enabled
   if ($require_alt) {
     $element['upload_container']['alt_wrapper']['details']['alt_text']['#required'] = TRUE;
   }
 }

    $element['#attached']['library'][] = 'dumi/dumi';
	$element['#element_submit'][] = [static::class, 'submitAltTextUpdates'];
    $element['#attached']['drupalSettings']['dumi'][$this->fieldDefinition->getName() . '_' . $delta] = [
      'fileSizeLimit' => $file_size_limit,
      'previewStyle' => $preview_style,
      'uploadUrl' => $upload_url,
      'csrfToken' => $csrf_token,
      'mediaBundle' => $media_bundle,
      'isMultiple' => $is_multiple,
      'cardinality' => $cardinality,
      'delta' => $delta,
      'fieldName' => $this->fieldDefinition->getName(),
      'hideEmptySlots' => $hide_empty_slots,
      'autoExpandSlots' => $auto_expand_slots,
    ];

    // Add proper cache metadata
    $element['#cache'] = [
      'contexts' => [
        'user.permissions',  // Vary by permissions (upload dumi files)
      ],
      'tags' => [
        'config:dumi.settings',  // Invalidate when config changes
      ],
    ];
    
    // Add media entity cache tag if media exists
    if (!empty($media_id)) {
      $element['#cache']['tags'][] = 'media:' . $media_id;
    }

    $element['#element_validate'][] = [$this, 'validateFileSize'];

    return $element;
  }

  /**
   * {@inheritdoc}
   * 
   * This follows Drupal's EXACT pattern for AJAX to work properly.
   * The wrapper ID must be on a container element, and the AJAX callback must return that container.
   */
  /**
   * {@inheritdoc}
   * 
   * Override formMultipleElements to wrap the table in a container
   * and add the "Remove All" button BELOW the table next to "Add another item".
   */
  protected function formMultipleElements(FieldItemListInterface $items, array &$form, FormStateInterface $form_state) {
  $field_name = $this->fieldDefinition->getName();
  $cardinality = $this->fieldDefinition->getFieldStorageDefinition()->getCardinality();
  $parents = $form['#parents'];

  // Determine how many widgets to display
  if ($cardinality == -1) {
    $field_state = static::getWidgetState($parents, $field_name, $form_state);
    $max = $field_state['items_count'];
    
    if (!$form_state->isRebuilding()) {
      $existing_count = count($items);
      $empty_slots = (int) $this->getSetting('empty_slots_for_unlimited') ?: 10;
      $max = max($existing_count + $empty_slots, $empty_slots);
    }
    
    $is_multiple = TRUE;
  }
  elseif ($cardinality > 1) {
    $max = $cardinality - 1;
    $is_multiple = TRUE;
  }
  else {
    $max = 0;
    $is_multiple = FALSE;
  }

  $title = $this->fieldDefinition->getLabel();
  $description = $this->getFilteredDescription();

  $elements = [];

  // Build elements with proper hidden row classes
  for ($delta = 0; $delta <= $max; $delta++) {
    if ($is_multiple) {
      $element = [
        '#title' => $this->t('@title (value @number)', ['@title' => $title, '@number' => $delta + 1]),
        '#title_display' => 'invisible',
        '#description' => '',
      ];
    }
    else {
      $element = [
        '#title' => $title,
        '#title_display' => 'before',
        '#description' => $description,
      ];
    }

    $element = $this->formSingleElement($items, $delta, $element, $form, $form_state);

    if ($element) {
      if ($is_multiple) {
        $element['_weight'] = [
          '#type' => 'weight',
          '#title' => $this->t('Weight for row @number)', ['@number' => $delta + 1]),
          '#title_display' => 'invisible',
          '#delta' => $max,
          '#default_value' => isset($items[$delta]->_weight) ? $items[$delta]->_weight : $delta,
          '#weight' => 100,
        ];
        
        // Check if this row should be hidden on initial load
        $field_item = $items->get($delta);
        $has_value = FALSE;
        if ($field_item) {
          $value = $field_item->getValue();
          $has_value = !empty($value['target_id']);
        }
        
        $hide_empty = $this->getSetting('hide_empty_slots') ?? TRUE;
        
        // Add hidden class for empty rows (except first row)
        if ($hide_empty && $delta > 0 && !$has_value) {
          if (!isset($element['#attributes'])) {
            $element['#attributes'] = [];
          }
          if (!isset($element['#attributes']['class'])) {
            $element['#attributes']['class'] = [];
          }
          
          $element['#attributes']['class'][] = 'dumi-hidden-row';
          $element['#attributes']['class'][] = 'dumi-row-force-hidden';
          
          if (isset($element['upload_container'])) {
            if (!isset($element['upload_container']['#attributes'])) {
              $element['upload_container']['#attributes'] = [];
            }
            if (!isset($element['upload_container']['#attributes']['class'])) {
              $element['upload_container']['#attributes']['class'] = [];
            }
            if (!in_array('dumi-hidden-slot', $element['upload_container']['#attributes']['class'])) {
              $element['upload_container']['#attributes']['class'][] = 'dumi-hidden-slot';
            }
          }
        }
      }

      $elements[$delta] = $element;
    }
  }

  if ($elements) {
    // Generate unique wrapper ID for AJAX
    $id_prefix = implode('-', array_merge($parents, [$field_name]));
    $wrapper_id = Html::getUniqueId($id_prefix . '-add-more-wrapper');
    
    // CRITICAL: Create wrapper container that holds everything
    $wrapper = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['field--widget-dumi-widget'],
        'id' => $wrapper_id,
        'data-field-name' => $field_name,
      ],
    ];

    // 1. Add progress bar FIRST (appears above table)
    $wrapper['progress_bar'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['dumi-multi-status', 'hidden']],
      '#weight' => -100,
      'progress_text' => [
        '#markup' => '<div class="dumi-progress-text">' . $this->t('Preparing upload...') . '</div>',
      ],
      'progress_bar' => [
        '#markup' => '<div class="dumi-progress-bar"><div class="dumi-progress-fill"></div></div>',
      ],
    ];

    // 2. Widget-level message container
    $wrapper['messages'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['dumi-widget-messages', 'hidden']],
      '#weight' => -90,
    ];

    // 3. Add the field elements with the table theme
    $wrapper['field_elements'] = $elements + [
      '#theme' => 'field_multiple_value_form',
      '#field_name' => $field_name,
      '#cardinality' => $cardinality,
      '#cardinality_multiple' => $is_multiple,
      '#required' => $this->fieldDefinition->isRequired(),
      '#title' => $title,
      '#description' => $description,
      '#max_delta' => $max,
      '#weight' => 0,
    ];

    // 4. Add 'add more' button INSIDE field_elements
    if ($cardinality == -1 && !$form_state->isProgrammed()) {
      $wrapper['field_elements']['add_more'] = [
        '#type' => 'submit',
        '#name' => strtr($id_prefix, '-', '_') . '_add_more',
        '#value' => $this->t('Add another item'),
        '#attributes' => ['class' => ['field-add-more-submit', 'dumi-ajax-add-more']],
        '#limit_validation_errors' => [array_merge($parents, [$field_name])],
        '#submit' => [[get_class($this), 'addMoreSubmit']],
        '#ajax' => [
          'callback' => [get_class($this), 'addMoreAjax'],
          'wrapper' => $wrapper_id,
          'effect' => 'fade',
        ],
        '#weight' => 1000,
      ];
    }

    // 5. CRITICAL: Add "Remove All" button as SIBLING to field_elements (not child)
    // This ensures it renders AFTER/BELOW the table, not inside it
    if ($is_multiple) {
      $wrapper['remove_all_container'] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['dumi-remove-all-wrapper'],
        ],
        '#weight' => 10, // Place after field_elements (which has weight 0)
      ];
      
      $wrapper['remove_all_container']['button'] = [
        '#type' => 'button',
        '#value' => $this->t('✕ Remove All Images'),
        '#attributes' => [
          'type' => 'button',
          'class' => ['dumi-remove-all-button', 'button', 'button--danger'],
          'data-field-name' => $field_name,
          'style' => 'padding: 8px 16px; background-color: #dc3545; color: white; border: 1px solid #c82333; border-radius: 4px; cursor: pointer; font-size: 14px; font-weight: 500; display: none; margin-top: 10px;',
        ],
      ];
    }

    return $wrapper;
    }

    return $elements;
  }
   
  /**
  * Ajax callback for the "Add another item" button.
  */
  public static function addMoreAjax(array $form, FormStateInterface $form_state) {
  $button = $form_state->getTriggeringElement();
  
  // Button is a direct child of the elements array
  $element_parents = array_slice($button['#array_parents'], 0, -1);
  $element = NestedArray::getValue($form, $element_parents);
  
  return $element;
  }

  /**
   * Validate file size.
   */
  public function validateFileSize(array &$element, FormStateInterface $form_state, array &$complete_form) {
    $limit_mb = (int) $this->getSetting('file_size_limit_mb');
    $config_limit = (int) ($this->configFactory->get('dumi.settings')->get('file_size_limit') ?: 5);
    if ($limit_mb <= 0) {
      $limit_mb = $config_limit;
    }
    if ($limit_mb <= 0) {
      return;
    }
    $limit_bytes = $limit_mb * 1024 * 1024;

    // Get the target_id from the current element being validated
    $target_id = NULL;
    if (isset($element['upload_container']['target_id']['#value'])) {
      $target_id = $element['upload_container']['target_id']['#value'];
    }
    
    if (empty($target_id)) {
      return;
    }
    
    try {
      $media = $this->entityTypeManager->getStorage('media')->load($target_id);
      if ($media) {
        $image_field_names = ['field_media_image', 'field_image', 'image'];
        foreach ($image_field_names as $field_name_check) {
          if ($media->hasField($field_name_check) && !$media->get($field_name_check)->isEmpty()) {
            $file = $media->get($field_name_check)->entity;
            if ($file && $file->getSize() > $limit_bytes) {
              $form_state->setError($element, $this->t('The uploaded image exceeds the file size limit of @limit MB.', [
                '@limit' => $limit_mb
              ]));
            }
            break;
          }
        }
      }
    } catch (\Exception $e) {
      $this->loggerFactory->get('dumi')->error('Error validating file size: @message', [
        '@message' => $e->getMessage()
      ]);
    }
  }

  protected function getBundleFromContext(array $form, FormStateInterface $form_state) {
    $form_object = $form_state->getFormObject();

    if (method_exists($form_object, 'getEntity')) {
      $entity = $form_object->getEntity();
      if ($entity && method_exists($entity, 'bundle')) {
        return $entity->bundle();
      }
    }

    if (isset($form['#bundle'])) {
      return $form['#bundle'];
    }

    $target_bundles = $this->getFieldSetting('handler_settings')['target_bundles'] ?? [];
    if (!empty($target_bundles) && count($target_bundles) === 1) {
      return reset($target_bundles);
    }

    return NULL;
  }

  protected function getCustomLabelContent($bundle, $config, $file_uri, $file_url) {
    if ($bundle) {
      $custom_label_fid = $config->get('labels.' . $bundle);
      if ($custom_label_fid) {
        try {
          $file = $this->entityTypeManager->getStorage('file')->load($custom_label_fid);
          if ($file) {
            $custom_file_uri = $file->getFileUri();
            $custom_file_url = $this->fileUrlGenerator->generateString($custom_file_uri);
            return '<img src="' . $custom_file_url . '" alt="' . $this->t('Upload button') . '" class="dumi-custom-label-image" />';
          }
        } catch (\Exception $e) {
          $this->loggerFactory->get('dumi')->warning('Failed to load custom label image for bundle @bundle: @message', [
            '@bundle' => $bundle,
            '@message' => $e->getMessage(),
          ]);
        }
      }
    }

    return (string) $this->t('Click to upload image');
  }


  /**
   * Massages the form values into the correct format for saving.
   * 
   * CRITICAL: With the wrapper structure, values come nested under 'field_elements'.
   * We also filter out non-field elements (buttons, containers, etc.).
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    $massaged_values = [];
    $alt_text_updates = [];
    
    // If values are wrapped in 'field_elements', extract them
    if (isset($values['field_elements'])) {
      $values = $values['field_elements'];
    }
    
    foreach ($values as $delta => $value) {
      // Skip non-numeric deltas (utility elements like 'add_more', 'remove_all_container', 'progress_bar', 'messages')
      if (!is_numeric($delta)) {
        continue;
      }
      
      // Handle double-nested structure from formMultipleElements
      if (isset($value[0]) && is_array($value[0])) {
        $value = $value[0];
      }
      
      $target_id = NULL;
      
      // Handle the structure where target_id is inside upload_container
      if (isset($value['upload_container']['target_id'])) {
        $target_id = $value['upload_container']['target_id'];
      }
      // Fallback to direct structure
      elseif (isset($value['target_id'])) {
        $target_id = $value['target_id'];
      }
      
      // Convert to integer and validate
      if (!empty($target_id)) {
        $target_id = (int) $target_id;
        
        // Only include non-empty values
        if ($target_id > 0) {
          // Store the media reference
          $massaged_values[] = ['target_id' => $target_id];
          
          // Collect alt text updates to process AFTER save
          $alt_text = NULL;
          if (isset($value['upload_container']['alt_wrapper']['details']['alt_text'])) {
            $alt_text = trim($value['upload_container']['alt_wrapper']['details']['alt_text']);
            if (!empty($alt_text)) {
              $alt_text_updates[$target_id] = $alt_text;
            }
          }
        }
      }
    }
    
    // Store alt text updates in form state for processing after entity save
    if (!empty($alt_text_updates)) {
      $form_state->set('dumi_alt_text_updates', $alt_text_updates);
    }
    
    return $massaged_values;
  }

 /**
 * Submit handler to update alt text after entity is saved.
 */
  public static function submitAltTextUpdates(array $form, FormStateInterface $form_state) {
  $alt_text_updates = $form_state->get('dumi_alt_text_updates');
  
  if (empty($alt_text_updates)) {
    return;
  }
  
  $entity_type_manager = \Drupal::entityTypeManager();
  $logger = \Drupal::logger('dumi');
  
  foreach ($alt_text_updates as $media_id => $alt_text) {
    try {
      $media = $entity_type_manager->getStorage('media')->load($media_id);
      if ($media && $media->access('update')) {
        $updated = FALSE;
        $image_field_names = ['field_media_image', 'field_image', 'image'];
        
        foreach ($image_field_names as $field_name_check) {
          if ($media->hasField($field_name_check) && !$media->get($field_name_check)->isEmpty()) {
            // Get current field value
            $field_value = $media->get($field_name_check)->getValue();
            
            if (!empty($field_value[0])) {
              // Update alt text in the field value array
              $field_value[0]['alt'] = $alt_text;
              
              // Set the entire field value back
              $media->set($field_name_check, $field_value);
              
              // Also update thumbnail alt if it exists
              if ($media->hasField('thumbnail') && !$media->get('thumbnail')->isEmpty()) {
                $thumb_value = $media->get('thumbnail')->getValue();
                if (!empty($thumb_value[0])) {
                  $thumb_value[0]['alt'] = $alt_text;
                  $media->set('thumbnail', $thumb_value);
                }
              }
              
              $media->save();
              $updated = TRUE;
              
              $logger->info('Updated alt text for media @id to: @alt', [
                '@id' => $media_id,
                '@alt' => $alt_text,
              ]);
              break;
            }
          }
        }
        
        if (!$updated) {
          $logger->warning('Could not update alt text for media @id - no suitable field found', [
            '@id' => $media_id,
          ]);
        }
      }
    } catch (\Exception $e) {
      $logger->error('Error updating alt text for media @id: @message', [
        '@id' => $media_id,
        '@message' => $e->getMessage(),
      ]);
    }
  }
 }

}