# DX Toolkit

Provides developer utilities, plugin systems, and service helpers to
improve the Drupal developer experience.

## Features

### ServiceInjector Plugin System

Automatically generates services using the factory pattern. Create a
plugin (optionally with a deriver), and services are auto-registered
during container compilation.

**Example:**
```yaml
services:
  my_module.my_service:
    class: Drupal\my_module\MyService
    arguments:
      - '@service_injector.node.storage'
      - '@service_injector.taxonomy_term.storage'
```

**Built-in Plugins:**
- **EntityStorage**: Generates storage handlers for all entity types
- **ConfigFactory**: Generates editable config objects

See SERVICEINJECTOR_EXAMPLE.md for complete documentation.

### EntityGenerator Plugin System

Plugin system for programmatically creating configuration entities with
derivative support.

### Utility Classes

- **Color**: Color manipulation utilities (hex normalization, luminance,
  contrast checking, YIQ calculations)
- **Json**: JSON utilities with control character removal
- **ArrayUtilities**: Array manipulation helpers
- **OptionsGenerator**: Generate form options from entities

### Base Classes & Interfaces

- **ServiceInstanceInterface**: Service locator pattern interface
- **ExtensionInterface**: Extension path and module utilities
- **StateInterface**: Enhanced state management
- **PluginBase**: Enhanced plugin base classes

## Requirements

- PHP 8.1 or higher
- Drupal 9, 10, or 11

## Installation

Install via Composer:

```bash
composer require drupal/dx_toolkit
```

Enable the module:

```bash
drush en dx_toolkit
```

## Usage

### ServiceInjector Example

Create a plugin:

```php
<?php

namespace Drupal\my_module\Plugin\ServiceInjector;

use Drupal\dx_toolkit\Plugin\ServiceInjectorBase;

/**
 * @ServiceInjector(
 *   id = "my_custom_injector",
 *   label = @Translation("My Custom Injector"),
 *   factoryService = "@my_module.factory",
 *   factoryMethod = "createInstance",
 *   factoryArguments = ["my_type"],
 *   serviceClass = "Drupal\my_module\MyInterface",
 *   servicePrefix = "service_injector",
 *   serviceSuffix = "custom"
 * )
 */
class MyCustomInjector extends ServiceInjectorBase {}
```

After cache rebuild, service `service_injector.my_type.custom` is
automatically available.

### Using Color Utilities

```php
use Drupal\dx_toolkit\Color;

// Normalize hex color
$color = Color::normalize('#abc');  // Returns '#aabbcc'

// Calculate luminance difference
$diff = Color::luminanceDifference('#000000', '#ffffff');

// Get best contrasting color
$best = Color::bestContrast('#336699', ['#000000', '#ffffff']);
```

### Using ServiceInstanceInterface

```php
namespace Drupal\my_module;

use Drupal\dx_toolkit\ServiceInstanceInterface;

interface MyServiceInterface extends ServiceInstanceInterface {
  public static function getServiceName(): string;
}

class MyService implements MyServiceInterface {
  use \Drupal\dx_toolkit\ServiceInstanceTrait;

  public static function getServiceName(): string {
    return 'my_module.my_service';
  }
}

// Usage anywhere:
$service = MyService::getService();
```

## Demo Module

The `dx_toolkit_demo` submodule provides working examples of all
plugin systems. Enable it to explore implementations:

```bash
drush en dx_toolkit_demo
```

See `modules/dx_toolkit_demo/README.md` for details.

## Development

### Running Tests

Quick start:
```bash
# Run all tests
./web/modules/contrib/dx_toolkit/run-tests.sh

# Run only unit tests
./web/modules/contrib/dx_toolkit/run-tests.sh unit

# Run only kernel tests
./web/modules/contrib/dx_toolkit/run-tests.sh kernel

# Generate coverage report
./web/modules/contrib/dx_toolkit/run-tests.sh coverage
```

For detailed testing instructions, see [TESTING.md](TESTING.md).

### Code Standards

The module follows strict mandate compliance (100% P0, 95% P1) which
exceeds standard Drupal coding standards. All code uses:
- Strict type declarations (`declare(strict_types=1);`)
- Proper dependency injection (no `\Drupal::service()` in classes)
- Functional programming patterns (array_map, array_reduce)
- 80-character line limits
- 2-space indentation

## Documentation

- [ServiceInjector Examples](SERVICEINJECTOR_EXAMPLE.md)
- [Demo Module Documentation](modules/dx_toolkit_demo/README.md)
- [Demo Comparison](modules/dx_toolkit_demo/COMPARISON.md)

## Maintainers

- Beau Townsend - [CoopWebBuilder](https://www.coopwebbuilder.com)

## License

GPL-2.0-or-later
