# ServiceInjector vs Traditional: Side-by-Side Comparison

This document shows real code comparisons demonstrating the benefits of ServiceInjector.

## Example: Content Auditor Service

### Traditional Approach

**ContentAuditor.php:**
```php
<?php

namespace Drupal\my_module;

use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;

class ContentAuditor {

  /**
   * Node storage handler.
   */
  protected EntityStorageInterface $nodeStorage;

  /**
   * Taxonomy term storage handler.
   */
  protected EntityStorageInterface $termStorage;

  /**
   * User storage handler.
   */
  protected EntityStorageInterface $userStorage;

  /**
   * Site configuration.
   */
  protected Config $siteConfig;

  /**
   * Constructs a ContentAuditor object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface
   *   $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface
   *   $configFactory
   *   The config factory.
   */
  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    ConfigFactoryInterface $configFactory,
  ) {
    // Manual initialization - verbose and error-prone
    $this->nodeStorage = $entityTypeManager->getStorage('node');
    $this->termStorage = $entityTypeManager->getStorage(
      'taxonomy_term'
    );
    $this->userStorage = $entityTypeManager->getStorage('user');
    $this->siteConfig = $configFactory->getEditable('system.site');
  }

  public function auditSite(): array {
    // Implementation...
  }
}
```

**my_module.services.yml:**
```yaml
services:
  my_module.content_auditor:
    class: Drupal\my_module\ContentAuditor
    arguments:
      - '@entity_type.manager'
      - '@config.factory'
```

**Lines of code: 53**

**Issues:**
- ❌ 4 property declarations
- ❌ Verbose constructor
- ❌ Manual getStorage() calls in constructor body
- ❌ Manual getEditable() call
- ❌ No type hints on $entityTypeManager parameter
- ❌ Easy to typo entity type IDs ('taxonmy_term' vs 'taxonomy_term')
- ❌ Constructor body can throw exceptions
- ❌ Harder to test (must mock EntityTypeManagerInterface)
- ❌ Not self-documenting (what storage handlers does this use?)

---

### ServiceInjector Approach

**ContentAuditor.php:**
```php
<?php

declare(strict_types=1);

namespace Drupal\my_module;

use Drupal\Core\Config\Config;
use Drupal\Core\Entity\EntityStorageInterface;

class ContentAuditor {

  /**
   * Constructs a ContentAuditor object.
   *
   * @param \Drupal\Core\Entity\EntityStorageInterface $nodeStorage
   *   Node storage from service_injector.node.storage.
   * @param \Drupal\Core\Entity\EntityStorageInterface $termStorage
   *   Term storage from service_injector.taxonomy_term.storage.
   * @param \Drupal\Core\Entity\EntityStorageInterface $userStorage
   *   User storage from service_injector.user.storage.
   * @param \Drupal\Core\Config\Config $siteConfig
   *   Site config from service_injector.system.site.config.
   */
  public function __construct(
    protected EntityStorageInterface $nodeStorage,
    protected EntityStorageInterface $termStorage,
    protected EntityStorageInterface $userStorage,
    protected Config $siteConfig,
  ) {}

  public function auditSite(): array {
    // Same implementation...
  }
}
```

**my_module.services.yml:**
```yaml
services:
  my_module.content_auditor:
    class: Drupal\my_module\ContentAuditor
    arguments:
      - '@service_injector.node.storage'
      - '@service_injector.taxonomy_term.storage'
      - '@service_injector.user.storage'
      - '@service_injector.system.site.config'
```

**Lines of code: 29 (45% reduction)**

**Benefits:**
- ✅ Clean constructor with proper type hints
- ✅ Protected properties via constructor promotion
- ✅ No manual initialization code
- ✅ Services injected ready-to-use
- ✅ Self-documenting dependencies
- ✅ Easy to test (mock individual storage handlers)
- ✅ Can't typo service IDs (IDE autocomplete)
- ✅ Compile-time validation

---

## Example: Form Using Multiple Entity Types

### Traditional Approach

**MyForm.php:**
```php
<?php

namespace Drupal\my_module\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

class MyForm extends FormBase {

  protected $nodeStorage;
  protected $termStorage;
  protected $userStorage;

  public function __construct(
    EntityTypeManagerInterface $entityTypeManager
  ) {
    $this->nodeStorage = $entityTypeManager->getStorage('node');
    $this->termStorage = $entityTypeManager->getStorage(
      'taxonomy_term'
    );
    $this->userStorage = $entityTypeManager->getStorage('user');
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  public function getFormId() {
    return 'my_module_my_form';
  }

  public function buildForm(
    array $form,
    FormStateInterface $form_state
  ) {
    $form['node'] = [
      '#type' => 'select',
      '#title' => $this->t('Node'),
      '#options' => $this->getNodeOptions(),
    ];

    $form['term'] = [
      '#type' => 'select',
      '#title' => $this->t('Term'),
      '#options' => $this->getTermOptions(),
    ];

    return $form;
  }

  protected function getNodeOptions(): array {
    $nodes = $this->nodeStorage->loadMultiple();
    return array_map(fn($n) => $n->label(), $nodes);
  }

  protected function getTermOptions(): array {
    $terms = $this->termStorage->loadMultiple();
    return array_map(fn($t) => $t->label(), $terms);
  }

  public function submitForm(
    array &$form,
    FormStateInterface $form_state
  ) {
    // Implementation...
  }
}
```

**Lines of code: 69**

---

### ServiceInjector Approach

**MyForm.php:**
```php
<?php

declare(strict_types=1);

namespace Drupal\my_module\Form;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

class MyForm extends FormBase {

  public function __construct(
    protected EntityStorageInterface $nodeStorage,
    protected EntityStorageInterface $termStorage,
  ) {}

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('service_injector.node.storage'),
      $container->get('service_injector.taxonomy_term.storage'),
    );
  }

  public function getFormId() {
    return 'my_module_my_form';
  }

  public function buildForm(
    array $form,
    FormStateInterface $form_state
  ) {
    $form['node'] = [
      '#type' => 'select',
      '#title' => $this->t('Node'),
      '#options' => $this->getNodeOptions(),
    ];

    $form['term'] = [
      '#type' => 'select',
      '#title' => $this->t('Term'),
      '#options' => $this->getTermOptions(),
    ];

    return $form;
  }

  protected function getNodeOptions(): array {
    $nodes = $this->nodeStorage->loadMultiple();
    return array_map(fn($n) => $n->label(), $nodes);
  }

  protected function getTermOptions(): array {
    $terms = $this->termStorage->loadMultiple();
    return array_map(fn($t) => $t->label(), $terms);
  }

  public function submitForm(
    array &$form,
    FormStateInterface $form_state
  ) {
    // Implementation...
  }
}
```

**Lines of code: 64 (7% reduction)**

Less dramatic in Forms because they still need `create()` method, but cleaner constructor.

---

## Testing Comparison

### Traditional Mocking

```php
public function testAuditSite() {
  $nodeStorage = $this->createMock(EntityStorageInterface::class);
  $nodeStorage->method('getQuery')
    ->willReturn($this->createMockQuery(158));

  $termStorage = $this->createMock(EntityStorageInterface::class);
  $termStorage->method('getQuery')
    ->willReturn($this->createMockQuery(88));

  $userStorage = $this->createMock(EntityStorageInterface::class);
  $userStorage->method('getQuery')
    ->willReturn($this->createMockQuery(2));

  $config = $this->createMock(Config::class);
  $config->method('get')->with('name')
    ->willReturn('Test Site');

  // Must mock EntityTypeManager
  $entityTypeManager = $this->createMock(
    EntityTypeManagerInterface::class
  );
  $entityTypeManager->method('getStorage')
    ->willReturnMap([
      ['node', $nodeStorage],
      ['taxonomy_term', $termStorage],
      ['user', $userStorage],
    ]);

  // Must mock ConfigFactory
  $configFactory = $this->createMock(
    ConfigFactoryInterface::class
  );
  $configFactory->method('getEditable')
    ->with('system.site')
    ->willReturn($config);

  $auditor = new ContentAuditor(
    $entityTypeManager,
    $configFactory
  );

  $stats = $auditor->auditSite();
  $this->assertEquals(158, $stats['node_count']);
}
```

### ServiceInjector Mocking

```php
public function testAuditSite() {
  $nodeStorage = $this->createMock(EntityStorageInterface::class);
  $nodeStorage->method('getQuery')
    ->willReturn($this->createMockQuery(158));

  $termStorage = $this->createMock(EntityStorageInterface::class);
  $termStorage->method('getQuery')
    ->willReturn($this->createMockQuery(88));

  $userStorage = $this->createMock(EntityStorageInterface::class);
  $userStorage->method('getQuery')
    ->willReturn($this->createMockQuery(2));

  $config = $this->createMock(Config::class);
  $config->method('get')->with('name')
    ->willReturn('Test Site');

  // Direct injection - no intermediate mocks needed!
  $auditor = new ContentAuditor(
    $nodeStorage,
    $termStorage,
    $userStorage,
    $config
  );

  $stats = $auditor->auditSite();
  $this->assertEquals(158, $stats['node_count']);
}
```

**50% fewer lines of test code** - no need to mock factories!

---

## Summary

| Aspect | Traditional | ServiceInjector | Winner |
|--------|------------|-----------------|--------|
| **Code Lines** | More verbose | 30-45% less | ✅ ServiceInjector |
| **Type Safety** | Limited | Full type hints | ✅ ServiceInjector |
| **Readability** | Verbose | Self-documenting | ✅ ServiceInjector |
| **Testability** | Complex mocks | Simple mocks | ✅ ServiceInjector |
| **Constructor** | Has logic | Declaration only | ✅ ServiceInjector |
| **IDE Support** | Limited | Full autocomplete | ✅ ServiceInjector |
| **Error Prone** | String typos | Compile-time check | ✅ ServiceInjector |
| **Performance** | Same runtime | Same runtime | 🟰 Tie |

ServiceInjector wins in every category except performance (where it ties).

**The bottom line:** ServiceInjector makes your code cleaner, safer, and more maintainable with zero runtime cost.
