<?php

declare(strict_types=1);

namespace Drupal\dx_toolkit_demo;

use Drupal\Core\Config\Config;
use Drupal\Core\Entity\EntityStorageInterface;

/**
 * Audits content across the site.
 *
 * This service demonstrates the utility of ServiceInjector by
 * showing how easily we can inject multiple entity storage
 * handlers without verbose factory calls.
 *
 * Compare this clean constructor to traditional approaches that
 * require injecting EntityTypeManagerInterface and calling
 * getStorage() for each entity type in the constructor body.
 */
class ContentAuditor implements ContentAuditorInterface
{

  /**
   * Constructs a ContentAuditor object.
   *
   * Notice how clean this is: each storage handler is directly
   * injected, ready to use, with proper type hints.
   *
   * @param \Drupal\Core\Entity\EntityStorageInterface $nodeStorage
   *   Node storage from service_injector.node.storage.
   * @param \Drupal\Core\Entity\EntityStorageInterface $termStorage
   *   Taxonomy term storage from
   *   service_injector.taxonomy_term.storage.
   * @param \Drupal\Core\Entity\EntityStorageInterface $userStorage
   *   User storage from service_injector.user.storage.
   * @param \Drupal\Core\Config\Config $siteConfig
   *   Site config from service_injector.system.site.config.
   */
  public function __construct(
    protected EntityStorageInterface $nodeStorage,
    protected EntityStorageInterface $termStorage,
    protected EntityStorageInterface $userStorage,
    protected Config $siteConfig,
  ) {}

  /**
   * @inheritDoc
   */
  public function auditSite(): array {
    $stats = [
      'site_name' => $this->siteConfig->get('name'),
      'node_count' => $this->countEntities($this->nodeStorage),
      'term_count' => $this->countEntities($this->termStorage),
      'user_count' => $this->countEntities($this->userStorage),
    ];

    $stats['total_content'] = $stats['node_count']
      + $stats['term_count']
      + $stats['user_count'];

    return $stats;
  }

  /**
   * @inheritDoc
   */
  public function auditByType(string $entity_type_id): array {
    $storage = match ($entity_type_id) {
      'node' => $this->nodeStorage,
      'taxonomy_term' => $this->termStorage,
      'user' => $this->userStorage,
      default => throw new \InvalidArgumentException(
        "Unsupported entity type: {$entity_type_id}"
      ),
    };

    $query = $storage->getQuery()
      ->accessCheck(FALSE);

    $ids = $query->execute();
    $entities = $storage->loadMultiple($ids);

    return [
      'entity_type' => $entity_type_id,
      'total_count' => count($entities),
      'entities' => array_map(
        fn ($entity) => [
          'id' => $entity->id(),
          'label' => $entity->label(),
          'bundle' => $entity->bundle(),
        ],
        array_slice($entities, 0, 10)
      ),
    ];
  }

  /**
   * Counts entities in storage.
   *
   * @param \Drupal\Core\Entity\EntityStorageInterface $storage
   *   The storage handler.
   *
   * @return int
   *   Entity count.
   */
  protected function countEntities(
    EntityStorageInterface $storage
  ): int {
    return (int) $storage->getQuery()
      ->accessCheck(FALSE)
      ->count()
      ->execute();
  }

}
