<?php

declare(strict_types=1);

namespace Drupal\dx_toolkit;

/**
 * Class Color
 *  Implementation of the extended Color utility class.
 *
 * @package Drupal\dx_toolkit
 */
class Color extends \Drupal\Component\Utility\Color implements ColorInterface
{

  /**
   * Returns the hex color code for white.
   *
   * @return string
   */
  protected static function whiteHex(): string {
    return '#ffffff';
  }

  /**
   * Returns the hex color code for black.
   *
   * @return string
   */
  protected static function blackHex(): string {
    return '#000000';
  }

  /**
   * @inheritDoc
   */
  public static function hexToRgba(string $hex, float $opacity = 1): string {
    $hex = static::normalize($hex);
    $rgba = [
      ...static::hexToRgb($hex),
      floatval($opacity),
    ];
    return 'rgba(' . implode(', ', $rgba) . ')';
  }

  /**
   * @inheritDoc
   */
  public static function contrast50(string $hex): string {
    $hex = static::normalize($hex);
    return ( hexdec($hex) > 0xffffff/2 )
      ? static::blackHex()
      : static::whiteHex();
  }

  /**
   * @inheritDoc
   */
  public static function contrastYIQ(string $hex): string {
    $hex = ltrim(static::normalize($hex), '#');
    [
      'red' => $r,
      'green' => $g,
      'blue' => $b
    ] = static::colorComponents($hex);
    $yiq = (
      (hexdec($r)*299) + (hexdec($g)*587) + (hexdec($b)*114)
    ) / 1000;
    return ($yiq >= 128) ? static::blackHex() : static::whiteHex();
  }

  /**
   * @inheritDoc
   */
  public static function contrastLuminance(string $hex): string {
    $light_diff = static::luminanceDifference($hex, static::whiteHex());
    $dark_diff = static::luminanceDifference($hex, static::blackHex());
    return $light_diff > $dark_diff ? static::whiteHex(): static::blackHex();
  }

  /**
   * @inheritDoc
   */
  public static function luminance(string $hex): float {
    $hex = ltrim(static::normalize($hex), '#');
    $components = static::colorComponents($hex);
    $r = hexdec($components['red']);
    $g = hexdec($components['green']);
    $b = hexdec($components['blue']);
    return 0.2126 * pow($r/255, 2.2)
      + 0.7152 * pow($g/255, 2.2)
      + 0.0722 * pow($b/255, 2.2);
  }


  /**
   * @inheritDoc
   */
  public static function luminanceDifference(
    string $hex1,
    string $hex2
  ): float {
    $luminance1 = static::luminance($hex1);
    $luminance2 = static::luminance($hex2);
    return $luminance1 > $luminance2
      ? ($luminance1 + 0.05) / ($luminance2 + 0.05)
      : ($luminance2 + 0.05) / ($luminance1 + 0.05);
  }

  /**
   * @inheritDoc
   */
  public static function calculateBestContrast(
    string $hex,
    array $candidates
  ): ?string {
    if (count($candidates) < 1) {
      return NULL;
    }
    $differences = array_map(
      fn($candidate) => [
        'hex' => static::normalize($candidate),
        'diff' => static::luminanceDifference(
          $hex,
          static::normalize($candidate)
        ),
      ],
      $candidates
    );
    $best = array_reduce(
      $differences,
      fn($carry, $item) => (
        $item['diff'] > ($carry['diff'] ?? 0) ? $item : $carry
      ),
      ['hex' => static::normalize($hex), 'diff' => 0]
    );
    return $best['hex'];
  }


  /**
   * Return an array containing each of the red, green and blue color components
   * from a CSS hex color value.
   *
   * @param string $hex
   *
   * @return array
   */
  protected static function colorComponents(string $hex): array {
    return [
      'red' => substr($hex,0,2),
      'green' => substr($hex,2,2),
      'blue' => substr($hex,4,2),
    ];
  }

  /**
   * Performs normalization on CSS hex color values and throws an exception in
   * the event of invalid CSS hex color value.
   *
   * @param string $hex
   *
   * @return string
   */
  protected static function normalize(string $hex): string {
    $hex = static::normalizeHexLength($hex);
    if (!static::validateHex($hex)) {
      throw new \InvalidArgumentException( static::exceptionMessage($hex) );
    }
    return $hex;
  }

  /**
   * Returns an exception message for the provided variable.
   *
   * @param $var
   *  The variable for which to generate an exception message.
   *
   * @return string
   *  The exception message to return.
   */
  protected static function exceptionMessage($var): string {
    return "'{$var}' is not a valid hex value.";
  }

}
