<?php

declare(strict_types=1);

namespace Drupal\dx_toolkit;

/**
 * Interface ColorInterface
 *  Extends the core Color utility class in order to provide enhancements to
 *  color handling, including contrast calculations.
 */
interface ColorInterface
{

  /**
   * Returns a CSS hex color value as a CSS rgba() color value.
   *
   * @param string $hex
   *  The hex color to convert. Cna be supplied in #abc or #abc123 format.
   *
   * @param float $opacity
   *  The opacity to apply to the returned rgba() CSS color value.
   *
   * @return string
   *  The generated rgba() CSS color value.
   */
  public static function hexToRgba(
    string $hex,
    float $opacity = 1
  ): string;

  /**
   * Returns the CSS hex color value for either white or black, based on which
   * of the two provides the highest contrast against the provided CSS hex color
   * value using the "naive" 50% white algorithm.
   *
   * @see https://24ways.org/2010/calculating-color-contrast/
   *
   * @param string $hex
   *  the CSS color hex value to evaluate for contrast.
   *
   * @return string
   *  The CSS hex color value for the higher contrast color between white and
   *  black.
   */
  public static function contrast50(string $hex): string;

  /**
   * Returns the CSS hex color value for either white or black, based on which
   * of the two provides the highest contrast against the provided CSS hex color
   * value using the YIQ color contrast comparison algorithm, derived from
   * NTSC television industry standards.
   *
   * @see https://24ways.org/2010/calculating-color-contrast/
   *
   * @param string $hex
   *  the CSS color hex value to evaluate for contrast.
   *
   * @return string
   *  The CSS hex color value for the higher contrast color or white or black.
   */
  public static function contrastYIQ(string $hex): string;

  /**
   * Returns the CSS hex color value for either white or black, based on which
   * of the two provides the highest contrast against the provided CSS hex color
   * value using the luminance color contrast comparison algorithm.
   *
   * For RGB color comparisons, this is considered the most effective contrast
   * calculation.
   *
   * @see https://www.splitbrain.org/blog/2008-09/
   *   18-calculating_color_contrast_with_php
   * @see https://stackoverflow.com/a/9733420/6679439
   *
   * @param string $hex
   *  the CSS color hex value to evaluate for contrast.
   *
   * @return string
   *  The CSS hex color value for the higher contrast color or white or black.
   */
  public static function contrastLuminance(string $hex): string;

  /**
   * Calculate the luminance of a CSS hex color value.
   *
   * @param string $hex
   *  The CSS hex color value whose luminance should be calculated.
   *
   * @return float
   *  The calculated luminance. Higher values are brighter values.
   */
  public static function luminance(string $hex): float;

  /**
   * Calculates the difference in luminance between two colors, each expressed
   * as a CSS hex color value. A luminance value >= 5 is best for readability.
   *
   * @see https://www.splitbrain.org/blog/2008-09/
   *   18-calculating_color_contrast_with_php
   *
   * @param string $hex1
   *  The first CSS hex color value to compare.
   *
   * @param string $hex2
   *  The second CSS hex color value to compare.
   *
   * @return float
   *  The calculated luminance difference. Higher is better. Optimal luminance
   *  is achieved at values >= 5.
   */
  public static function luminanceDifference(string $hex1, string $hex2): float;

  /**
   * Returns the best CSS hex color value from an array of candidates as
   * compared in contrast to a given CSS hex color value.
   *
   * @param string $hex
   *  The CSS hex color value against which to compare all candidate colors.
   *
   * @param array $candidates
   *  An array of CSS hex color values to compare to the initially provided
   *  CSS hex color value.
   *
   * @return string|null
   *  The highest contrasting CSS hex color value from the provided list of
   *  candidate colors. If an empty array of candidates is provided, NULL is
   *  returned.
   */
  public static function calculateBestContrast(
    string $hex,
    array $candidates
  ): ?string;

}
