<?php

declare(strict_types=1);

namespace Drupal\dx_toolkit;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\DependencyInjection\ServiceProviderBase;
use Drupal\dx_toolkit\Annotation\ServiceInjector;
use Symfony\Component\DependencyInjection\Definition;
use Symfony\Component\DependencyInjection\Reference;

/**
 * Service provider for dx_toolkit module.
 *
 * Discovers ServiceInjector plugins during container compilation and
 * registers their generated services directly into the DI container.
 *
 * @package Drupal\dx_toolkit
 */
class DxToolkitServiceProvider extends ServiceProviderBase
{

  /**
   * @inheritDoc
   */
  public function register(ContainerBuilder $container): void {
    try {
      $definitions = $this->discoverServiceDefinitions($container);

      foreach ($definitions as $service_id => $config) {
        $this->registerServiceFromConfig(
          $container,
          $service_id,
          $config
        );
      }
    }
    catch (\Throwable $e) {
      error_log(
        "ServiceInjector discovery failed: {$e->getMessage()}\n"
        . $e->getTraceAsString()
      );
    }
  }

  /**
   * Discovers all service definitions from ServiceInjector plugins.
   *
   * @param \Drupal\Core\DependencyInjection\ContainerBuilder $container
   *   The container builder.
   *
   * @return array
   *   Array of service definitions keyed by service ID.
   */
  protected function discoverServiceDefinitions(
    ContainerBuilder $container
  ): array {
    $namespaces = $this->moduleNamespaces($container);
    $plugins = $this->scanForPlugins($namespaces);
    $definitions = [];

    foreach ($plugins as $plugin_id => $plugin_info) {
      $plugin_defs = $this->processPlugin($plugin_info, $container);
      $definitions = [...$definitions, ...$plugin_defs];
    }

    return $definitions;
  }

  /**
   * Returns module namespaces from container parameters.
   *
   * @param \Drupal\Core\DependencyInjection\ContainerBuilder $container
   *   The container builder.
   *
   * @return array
   *   Array of module namespaces keyed by module name.
   */
  protected function moduleNamespaces(
    ContainerBuilder $container
  ): array {
    $namespaces = [];

    if (!$container->hasParameter('container.modules')) {
      return $namespaces;
    }

    $modules = $container->getParameter('container.modules');

    foreach ($modules as $module => $info) {
      $namespaces[$module] = "Drupal\\{$module}";
    }

    return $namespaces;
  }

  /**
   * Scans for ServiceInjector plugin classes.
   *
   * @param array $namespaces
   *   Module namespaces keyed by module name.
   *
   * @return array
   *   Discovered plugins with 'class' and 'annotation' keys.
   */
  protected function scanForPlugins(array $namespaces): array {
    $plugins = [];

    foreach ($namespaces as $module => $namespace) {
      $path = $this->namespaceToPath($namespace);
      $plugin_path = "{$path}/Plugin/ServiceInjector";

      if (!is_dir($plugin_path)) {
        continue;
      }

      $files = glob("{$plugin_path}/*.php") ?: [];

      foreach ($files as $file) {
        if (!$class = $this->fileToClass($file, $namespace)) {
          continue;
        }

        if (!$annotation = $this->readAnnotation($class)) {
          continue;
        }

        $plugins[$annotation->id] = [
          'class' => $class,
          'annotation' => $annotation,
        ];
      }
    }

    return $plugins;
  }

  /**
   * Converts namespace to filesystem path.
   *
   * @param string $namespace
   *   The namespace.
   *
   * @return string
   *   The filesystem path.
   */
  protected function namespaceToPath(string $namespace): string {
    $loader = require DRUPAL_ROOT . '/autoload.php';
    $prefixes = $loader->getPrefixesPsr4();

    foreach ($prefixes as $prefix => $paths) {
      if (str_starts_with($namespace . '\\', $prefix)) {
        $relative = substr($namespace, strlen(rtrim($prefix, '\\')));
        return $paths[0] . str_replace('\\', '/', $relative);
      }
    }

    throw new \RuntimeException(
      "Could not resolve namespace: {$namespace}"
    );
  }

  /**
   * Converts file path to fully qualified class name.
   *
   * @param string $file
   *   The file path.
   * @param string $base_namespace
   *   The base namespace.
   *
   * @return string|null
   *   The class name or NULL if cannot be determined.
   */
  protected function fileToClass(
    string $file,
    string $base_namespace
  ): ?string {
    $basename = basename($file, '.php');

    if (empty($basename)) {
      return NULL;
    }

    return "{$base_namespace}\\Plugin\\ServiceInjector\\{$basename}";
  }

  /**
   * Reads ServiceInjector annotation from class.
   *
   * @param string $class
   *   The class name.
   *
   * @return \Drupal\dx_toolkit\Annotation\ServiceInjector|null
   *   The annotation or NULL if not found.
   */
  protected function readAnnotation(string $class): ?ServiceInjector {
    if (!class_exists($class)) {
      return NULL;
    }

    try {
      $reflection = new \ReflectionClass($class);
      $doc_comment = $reflection->getDocComment();

      if (!$doc_comment) {
        return NULL;
      }

      if (!preg_match('/@ServiceInjector\s*\(/s', $doc_comment)) {
        return NULL;
      }

      return $this->parseServiceInjectorAnnotation($doc_comment);
    }
    catch (\Throwable $e) {
      error_log(
        "Failed to read annotation for {$class}: {$e->getMessage()}"
      );
    }

    return NULL;
  }

  /**
   * Parses ServiceInjector annotation from docblock.
   *
   * @param string $doc_comment
   *   The docblock comment.
   *
   * @return \Drupal\dx_toolkit\Annotation\ServiceInjector|null
   *   The parsed annotation or NULL if cannot be parsed.
   */
  protected function parseServiceInjectorAnnotation(
    string $doc_comment
  ): ?ServiceInjector {
    $annotation = new ServiceInjector([]);

    preg_match('/id\s*=\s*"([^"]+)"/', $doc_comment, $matches);
    $annotation->id = $matches[1] ?? '';

    preg_match(
      '/label\s*=\s*@Translation\s*\(\s*"([^"]+)"\s*\)/i',
      $doc_comment,
      $matches
    );
    $annotation->label = $matches[1] ?? '';

    preg_match(
      '/description\s*=\s*@Translation\s*\(\s*"([^"]+)"\s*\)/i',
      $doc_comment,
      $matches
    );
    $annotation->description = $matches[1] ?? '';

    preg_match(
      '/factoryService\s*=\s*"([^"]+)"/',
      $doc_comment,
      $matches
    );
    $annotation->factoryService = $matches[1] ?? '';

    preg_match(
      '/factoryMethod\s*=\s*"([^"]+)"/',
      $doc_comment,
      $matches
    );
    $annotation->factoryMethod = $matches[1] ?? '';

    preg_match('/serviceClass\s*=\s*"([^"]+)"/', $doc_comment, $matches);
    $annotation->serviceClass = $matches[1] ?? '';

    preg_match(
      '/servicePrefix\s*=\s*"([^"]+)"/',
      $doc_comment,
      $matches
    );
    $annotation->servicePrefix = $matches[1] ?? 'service_injector';

    preg_match(
      '/serviceSuffix\s*=\s*"([^"]+)"/',
      $doc_comment,
      $matches
    );
    $annotation->serviceSuffix = $matches[1] ?? '';

    preg_match('/deriver\s*=\s*"([^"]+)"/', $doc_comment, $matches);
    $annotation->deriver = $matches[1] ?? '';

    if (empty($annotation->id)) {
      return NULL;
    }

    return $annotation;
  }

  /**
   * Processes a plugin to generate service definitions.
   *
   * @param array $plugin_info
   *   Plugin info with 'class' and 'annotation' keys.
   * @param \Drupal\Core\DependencyInjection\ContainerBuilder $container
   *   The container builder.
   *
   * @return array
   *   Service definitions keyed by service ID.
   */
  protected function processPlugin(
    array $plugin_info,
    ContainerBuilder $container
  ): array {
    try {
      $annotation = $plugin_info['annotation'];
      $this->validateAnnotation($annotation);

      if (empty($annotation->deriver)) {
        return $this->createServiceDefinition($annotation);
      }

      return $this->processDeriver(
        $annotation->deriver,
        $annotation,
        $container
      );
    }
    catch (\Throwable $e) {
      error_log(
        "Failed to process plugin '{$plugin_info['class']}': "
        . $e->getMessage()
      );
      return [];
    }
  }

  /**
   * Validates ServiceInjector annotation.
   *
   * @param \Drupal\dx_toolkit\Annotation\ServiceInjector $annotation
   *   The annotation to validate.
   *
   * @return $this
   *   Reference to self.
   *
   * @throws \LogicException
   *   If validation fails.
   */
  protected function validateAnnotation(
    ServiceInjector $annotation
  ): static {
    $required = [
      'id',
      'factoryService',
      'factoryMethod',
      'serviceClass',
    ];

    foreach ($required as $field) {
      if (empty($annotation->{$field})) {
        throw new \LogicException(
          "ServiceInjector plugin '{$annotation->id}' missing "
          . "required field: {$field}"
        );
      }
    }

    return $this;
  }

  /**
   * Creates service definition from annotation without deriver.
   *
   * @param \Drupal\dx_toolkit\Annotation\ServiceInjector $annotation
   *   The annotation.
   *
   * @return array
   *   Service definitions keyed by service ID.
   */
  protected function createServiceDefinition(
    ServiceInjector $annotation
  ): array {
    $definition = (array) $annotation;
    $service_id = $this->buildServiceId($definition);

    return [
      $service_id => [
        'class' => $annotation->serviceClass,
        'factory' => [
          $annotation->factoryService,
          $annotation->factoryMethod,
        ],
        'arguments' => $annotation->factoryArguments,
      ],
    ];
  }

  /**
   * Processes a deriver to generate derivative service definitions.
   *
   * @param string $deriver_class
   *   The deriver class name.
   * @param \Drupal\dx_toolkit\Annotation\ServiceInjector $annotation
   *   The base annotation.
   * @param \Drupal\Core\DependencyInjection\ContainerBuilder $container
   *   The container builder.
   *
   * @return array
   *   Service definitions keyed by service ID.
   */
  protected function processDeriver(
    string $deriver_class,
    ServiceInjector $annotation,
    ContainerBuilder $container
  ): array {
    if (!class_exists($deriver_class)) {
      error_log("Deriver class not found: {$deriver_class}");
      return [];
    }

    try {
      $deriver = new $deriver_class();
      $base_definition = (array) $annotation;
      $derivatives = $deriver->getDerivativeDefinitions(
        $base_definition,
        $container
      );

      $services = [];

      foreach ($derivatives as $derivative_id => $derivative_def) {
        $service_id = $this->buildServiceId($derivative_def);
        $services[$service_id] = [
          'class' => $derivative_def['serviceClass'],
          'factory' => [
            $derivative_def['factoryService'],
            $derivative_def['factoryMethod'],
          ],
          'arguments' => $derivative_def['factoryArguments'] ?? [],
        ];
      }

      return $services;
    }
    catch (\Throwable $e) {
      error_log(
        "Deriver {$deriver_class} failed: {$e->getMessage()}"
      );
      return [];
    }
  }

  /**
   * Builds service ID from definition.
   *
   * @param array $definition
   *   Service definition with servicePrefix, factoryArguments, serviceSuffix.
   *
   * @return string
   *   The service ID.
   */
  protected function buildServiceId(array $definition): string {
    $parts = [
      $definition['servicePrefix'] ?? 'service_injector',
    ];

    foreach ($definition['factoryArguments'] ?? [] as $arg) {
      $parts[] = $arg;
    }

    if (!empty($definition['serviceSuffix'])) {
      $parts[] = $definition['serviceSuffix'];
    }

    return implode('.', array_filter($parts));
  }

  /**
   * Registers a service from configuration array.
   *
   * @param \Drupal\Core\DependencyInjection\ContainerBuilder $container
   *   The container builder.
   * @param string $service_id
   *   The service ID.
   * @param array $config
   *   Service configuration with keys: class, factory, arguments.
   *
   * @return $this
   *   Reference to self.
   */
  protected function registerServiceFromConfig(
    ContainerBuilder $container,
    string $service_id,
    array $config
  ): static {
    try {
      $definition = new Definition(
        $config['class'],
        $config['arguments'] ?? []
      );

      $definition->setPublic(TRUE);

      if (isset($config['factory'])) {
        [$factory_service, $factory_method] = $config['factory'];

        if (
          is_string($factory_service)
          && str_starts_with($factory_service, '@')
        ) {
          $factory_service = new Reference(
            substr($factory_service, 1)
          );
        }

        $definition->setFactory([$factory_service, $factory_method]);
      }

      $container->setDefinition($service_id, $definition);
    }
    catch (\Throwable $e) {
      error_log(
        "ServiceInjector: Failed to register {$service_id}: "
        . $e->getMessage()
      );
    }

    return $this;
  }

}
