<?php

declare(strict_types=1);

namespace Drupal\dx_toolkit\Plugin;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityStorageException;

/**
 * Provides a plugin that generates a new entity bundle type as a companion
 * bundle to a source entity type. This is performed via derivers.
 *
 * Sample usages could be:
 * - Adding a geolocation field to every content type.
 * - Generating a view for each term in a given taxonomy vocabulary.
 * - Re-configuring the entity form display for all paragraph types.
 */
interface EntityGeneratorInterface extends PluginBaseInterface
{

  /**
   * Returns the name of the configuration file, without extension, that
   * contains the base configuration for newly generated entities.
   *
   * @return string
   */
  public function getConfigName(): string;

  /**
   * Returns the path that contains the source configuration file, relative to
   * the path of the source module.
   *
   * @return string
   */
  public function getConfigPath(): string;

  /**
   * The name of the module that contains the source configuration file.
   *
   * @return string
   */
  public function sourceModule(): string;

  /**
   * Returns the entity type ID that identifies the type of entity to be
   * generated.
   *
   * @return string
   */
  public function getTargetEntityTypeId(): string;

  /**
   * Returns the entity type ID that identifies the entity type to act as a
   * source for entity bundles.
   *
   * @return string
   */
  public function getSourceEntityTypeId(): string;

  /**
   * Returns the name of the bundle to reference when generating a new entity.
   *
   * @return string
   */
  public function bundleName(): string;

  /**
   * Returns an associative array that represents the contents of the base
   * configuration file.
   *
   * @return array
   */
  public function getBaseConfigurationData(): array;

  /**
   * Returns an array of configuration values that are unique to this
   * combination of source entity type and source bundle name.
   *
   * @return array
   */
  public function customConfiguration(): array;

  /**
   * Returns the merged results of combining base and custom configuration.
   * Newly generated entities will be generated from this configuration.
   *
   * @return array
   */
  public function getTargetEntityConfiguration(): array;

  /**
   * Returns the ID to assign to the newly generated entity.
   *
   * Note that IDs must be no longer than 32 characters in length.
   *
   * @return string
   */
  public function targetId(): string;

  /**
   * Performs the entity generation process.
   *
   * @throws EntityStorageException
   */
  public function generateEntity(): static;

  /**
   * Deletes the target entity, if it exists.
   *
   * @return $this
   * @throws EntityStorageException
   */
  public function deleteTargetEntity(): static;

  /**
   * Returns the entity that was generated by this plugin, or NULL if no
   * matching entity could be found.
   *
   * @return EntityInterface|NULL
   */
  public function getTargetEntity(): ?EntityInterface;

  /**
   * Returns TRUE if a target entity exists.
   *
   * @return bool
   */
  public function hasTargetEntity(): bool;

}
