<?php

declare(strict_types=1);

namespace Drupal\dx_toolkit\Plugin;

use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\dx_toolkit\Annotation\EntityGenerator;
use Traversable;

/**
 * Class EntityGeneratorManager
 *  Implements a factory to manage EntityGenerator plugins along with methods
 *  to bulk generate and delete companion entities.
 *
 * @package Drupal\dx_toolkit\Plugin
 */
class EntityGeneratorManager extends PluginManager
  implements PluginManagerInterface
{

  /**
   * @inheritDoc
   */
  public function __construct(
    Traversable $namespaces,
    CacheBackendInterface $cache_backend,
    ModuleHandlerInterface $module_handler
  ) {
    parent::__construct(
      'Plugin/EntityGenerator',
      $namespaces,
      $module_handler,
      EntityGeneratorInterface::class,
      EntityGenerator::class
    );
    $this->alterInfo('entity_generator_info');
    $this->setCacheBackend(
      $cache_backend,
      'entity_generator_plugins'
    );
  }

  /**
   * @inheritDoc
   */
  public static function getServiceName(): string {
    return 'plugin.manager.entity_generator';
  }

  /**
   * Generates an entity for every defined EntityGeneratorBase plugin.
   * Existing target entities will be skipped.
   *
   * @return $this
   * @throws PluginException|EntityStorageException
   */
  public function generateAllEntities(
    string $base_plugin_id
  ): EntityGeneratorManager {
    $instances = $this->derivativeInstances($base_plugin_id);
    array_walk(
      $instances,
      fn(EntityGeneratorInterface $instance) => $instance->generateEntity()
    );
    return $this;
  }

  /**
   * Returns an array of all derivative IDs for a provided base plugin ID.
   *
   * @param string $base_plugin_id
   *  The base plugin ID for which to return derivative plugin IDs
   *
   * @return array
   */
  protected function derivativeIds(string $base_plugin_id): array {
    return array_keys( $this->getPluginDerivatives($base_plugin_id) );
  }

  /**
   * Returns an array of all EntityGenerator plugin instances derived from
   * the provided base plugin ID.
   *
   * @param string $base_plugin_id
   *  The base plugin ID for which to return derivative plugin instances.
   *
   * @return EntityGeneratorInterface[]
   * @throws PluginException
   */
  protected function derivativeInstances(string $base_plugin_id): array {
    $derivative_ids = $this->derivativeIds($base_plugin_id);
    return $this->createInstances($derivative_ids);
  }

  /**
   * Returns an associative array of entities generated by the derived plugins
   * identified by the base plugin ID. Each entity will be keyed by its
   * derivative plugin ID.
   *
   * @param string $base_plugin_id
   *  The base plugin ID for which to return generated entities.
   *
   * @return array
   * @throws PluginException
   */
  public function entities(string $base_plugin_id): array {
    /**
     * @var string $plugin_id
     * @var EntityGeneratorInterface $instance
     */
    return array_filter(
      $this->derivativeInstances($base_plugin_id),
      fn (EntityGeneratorInterface $instance) => $instance->hasTargetEntity(),
    );
  }

  /**
   * Deletes all target entities generated by the derived plugins identified by
   * the base plugin ID.
   *
   * @param string $base_plugin_id
   *
   * @return $this
   * @throws PluginException
   * @throws EntityStorageException
   */
  public function deleteAllEntities(
    string $base_plugin_id
  ): EntityGeneratorManager {
    $instances = $this->derivativeInstances($base_plugin_id);
    array_walk(
      $instances,
      fn(EntityGeneratorInterface $instance) => (
        $instance->deleteTargetEntity()
      )
    );
    return $this;
  }

}
