<?php

declare(strict_types=1);

namespace Drupal\dx_toolkit\Plugin;

use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\feeds\FeedInterface;
use Drupal\feeds\FeedStorageInterface;
use Drupal\feeds\FeedTypeInterface;

/**
 * Class FeedTypeGeneratorBase
 *  A specialized EntityGeneratorBase plugin that provides additional
 *  functionality for generating Feeds when generating companion Feed Types.
 */
class FeedTypeGeneratorBase extends EntityGeneratorBase
  implements FeedTypeGeneratorInterface
{

  /**
   * @var FeedStorageInterface
   */
  protected FeedStorageInterface $feedStorage;

  /**
   * @inheritDoc
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    EntityTypeManagerInterface $entity_type_manager,
    EntityTypeBundleInfoInterface $entity_bundle_info,
    ModuleHandlerInterface $module_handler
  ) {
    parent::__construct(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $entity_type_manager,
      $entity_bundle_info,
      $module_handler
    );
    $this->feedStorage = $entity_type_manager
      ->getStorage('feeds_feed');
  }

  /**
   * @inheritDoc
   */
  public function customConfiguration(): array {
    return [
      'label' => $this->feedTypeLabel(),
      'id' => $this->targetId(),
      'description' => $this->feedTypeDescription(),
      'processor' => "entity:{$this->getSourceEntityTypeId()}",
      'processor_configuration' => [
        'values' => [
          'type' => $this->bundleName(),
        ],
      ],
    ];
  }

  /**
   * @inheritDoc
   */
  public function targetId(): string {
    return "{$this->bundleName()}_importer";
  }

  /**
   * Returns a label for the newly generated feed type.
   *
   * @return string
   */
  protected function feedTypeLabel(): string {
    return (string) $this->t('@entity_type @bundle importer', [
      '@entity_type' => (string) $this->sourceEntityDefinition->getLabel(),
      '@bundle' => $this->getSourceBundleLabel(),
    ]);
  }

  /**
   * Returns a description for the newly generated feed type.
   *
   * @return string
   */
  protected function feedTypeDescription(): string {
    return (string) $this->t('Import @entity_type @bundle entities', [
      '@entity_type' => (string) $this->sourceEntityDefinition->getLabel(),
      '@bundle' => $this->getSourceBundleLabel(),
    ]);
  }

  /**
   * @inheritDoc
   */
  public function generateFeed(string $source = ''): static {
    if ($this->feeds()) {
      return $this;
    }
    /** @var FeedTypeInterface $feed_type */
    $feed_type = $this->getTargetEntity();
    $feed = $this->feedStorage->create([
      'id' => "{$feed_type->id()}_feed",
      'title' => "{$feed_type->label()} feed",
      'type' => $feed_type->id(),
      'uid' => static::feedOwnerUid(),
    ]);
    $feed->setSource($source);
    $feed->save();
    return $this;
  }

  /**
   * Returns the UID of the feed owner. Override this method to assign a
   * different owner.
   *
   * @return string
   */
  protected static function feedOwnerUid(): string {
    return '1';
  }

  /**
   * @inheritDoc
   */
  public function feeds(): array {
    return $this->feedStorage
      ->loadByProperties( ['type' => $this->targetId()] ) ?? [];
  }

  /**
   * @inheritDoc
   */
  public function deleteFeeds(): static {
    if (empty($feeds = $this->feeds())) {
      return $this;
    }
    array_walk($feeds, fn(FeedInterface $feed) => $feed->delete());
    return $this;
  }

  /**
   * @inheritDoc
   */
  public function import(string $source = NULL): static {
    $feeds = $this->feeds();
    array_walk(
      $feeds,
      function(FeedInterface $feed) use ($source) {
        if ($source) {
          $feed->setSource($source);
        }
        $feed->import();
      }
    );
    return $this;
  }

}
