<?php

declare(strict_types=1);

namespace Drupal\dx_toolkit\Plugin\ServiceInjector\Derivative;

use Drupal\Component\Plugin\Derivative\DeriverBase;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\Discovery\ContainerDeriverInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Deriver for EntityStorage ServiceInjector plugin.
 *
 * Creates a derivative for each entity type, generating services like:
 * - service_injector.node.storage
 * - service_injector.user.storage
 * - service_injector.taxonomy_term.storage
 *
 * @package Drupal\dx_toolkit\Plugin\ServiceInjector\Derivative
 */
class EntityStorageDeriver extends DeriverBase
  implements ContainerDeriverInterface
{
  use StringTranslationTrait;

  /**
   * Constructs an EntityStorageDeriver object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface|null
   *   $entityTypeManager
   *   The entity type manager (optional, for runtime context).
   */
  public function __construct(
    protected ?EntityTypeManagerInterface $entityTypeManager = NULL
  ) {}

  /**
   * @inheritDoc
   */
  public static function create(
    ContainerInterface $container,
    $base_plugin_id
  ): static {
    return new static($container->get('entity_type.manager'));
  }

  /**
   * @inheritDoc
   */
  public function getDerivativeDefinitions(
    $base_definition,
    ?ContainerBuilder $container = NULL
  ): array {
    if ($container) {
      $entity_types = $this->discoverEntityTypesFromContainer(
        $container
      );
    }
    else {
      $entity_types = $this->discoverEntityTypesFromManager();
    }

    return $this->derivatives = array_map(
      fn ($entity_type) => [
        ...$base_definition,
        'label' => "{$entity_type['label']} Storage",
        'description' => (
          "Storage handler for {$entity_type['label']} entities."
        ),
        'factoryArguments' => [$entity_type['id']],
        'entityTypeId' => $entity_type['id'],
      ],
      $entity_types
    );
  }

  /**
   * Discovers entity types from EntityTypeManager (runtime context).
   *
   * @return array
   *   Entity type info arrays.
   */
  protected function discoverEntityTypesFromManager(): array {
    if (!$this->entityTypeManager) {
      return [];
    }

    return array_map(
      fn ($entity_type) => [
        'id' => $entity_type->id(),
        'label' => $entity_type->getLabel(),
      ],
      $this->entityTypeManager->getDefinitions()
    );
  }

  /**
   * Discovers entity types from container (compile-time context).
   *
   * Scans module namespaces for entity classes and reads their
   * annotations to discover entity type definitions.
   *
   * @param \Drupal\Core\DependencyInjection\ContainerBuilder $container
   *   The container builder.
   *
   * @return array
   *   Entity type info arrays.
   */
  protected function discoverEntityTypesFromContainer(
    ContainerBuilder $container
  ): array {
    $entity_types = [];
    $modules = $container->getParameter('container.modules');

    foreach ($modules as $module => $info) {
      $namespace = "Drupal\\{$module}";
      $entity_path = $this->namespaceToPath($namespace)
        . '/Entity';

      if (!is_dir($entity_path)) {
        continue;
      }

      $discovered = $this->scanEntityDirectory($entity_path, $namespace);
      $entity_types = [...$entity_types, ...$discovered];
    }

    return $entity_types;
  }

  /**
   * Scans entity directory for entity class files.
   *
   * @param string $path
   *   The directory path.
   * @param string $namespace
   *   The base namespace.
   *
   * @return array
   *   Discovered entity type info arrays.
   */
  protected function scanEntityDirectory(
    string $path,
    string $namespace
  ): array {
    $entity_types = [];
    $files = glob("{$path}/*.php") ?: [];

    foreach ($files as $file) {
      $basename = basename($file, '.php');
      $class = "{$namespace}\\Entity\\{$basename}";

      if (!class_exists($class)) {
        continue;
      }

      if (!$entity_info = $this->extractEntityInfo($class)) {
        continue;
      }

      $entity_types[$entity_info['id']] = $entity_info;
    }

    return $entity_types;
  }

  /**
   * Extracts entity type info from class annotation.
   *
   * @param string $class
   *   The entity class name.
   *
   * @return array|null
   *   Entity type info or NULL if not an entity.
   */
  protected function extractEntityInfo(string $class): ?array {
    try {
      $reflection = new \ReflectionClass($class);
      $doc_comment = $reflection->getDocComment();

      if (!$doc_comment) {
        return NULL;
      }

      if (
        !preg_match(
          '/@(ContentEntity|ConfigEntity)Type\s*\(/i',
          $doc_comment
        )
      ) {
        return NULL;
      }

      if (!preg_match('/id\s*=\s*"([^"]+)"/', $doc_comment, $matches)) {
        return NULL;
      }

      $id = $matches[1];

      preg_match(
        '/label\s*=\s*@Translation\s*\(\s*"([^"]+)"\s*\)/i',
        $doc_comment,
        $label_matches
      );
      $label = $label_matches[1] ?? ucfirst(str_replace('_', ' ', $id));

      return [
        'id' => $id,
        'label' => $label,
      ];
    }
    catch (\Throwable $e) {
      error_log(
        "Failed to extract entity info from {$class}: "
        . $e->getMessage()
      );
      return NULL;
    }
  }

  /**
   * Converts namespace to filesystem path.
   *
   * @param string $namespace
   *   The namespace.
   *
   * @return string
   *   The filesystem path.
   */
  protected function namespaceToPath(string $namespace): string {
    $loader = require DRUPAL_ROOT . '/autoload.php';
    $prefixes = $loader->getPrefixesPsr4();

    foreach ($prefixes as $prefix => $paths) {
      if (str_starts_with($namespace . '\\', $prefix)) {
        $relative = substr($namespace, strlen(rtrim($prefix, '\\')));
        return $paths[0] . str_replace('\\', '/', $relative);
      }
    }

    throw new \RuntimeException(
      "Could not resolve namespace: {$namespace}"
    );
  }

}
