<?php

declare(strict_types=1);

namespace Drupal\dx_toolkit;

use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Class StateBase
 *  Provides an implementation of the StateInterface wrapper for the StateBase
 *  key/value store.
 *
 * @package Drupal\dx_toolkit
 */
abstract class StateBase implements StateInterface
{
  use AsArrayTrait;
  use StringTranslationTrait;

  /**
   * Constructs a StateBase object.
   *
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   */
  public function __construct(
    protected \Drupal\Core\State\StateInterface $state,
  ) {}

  /**
   * Returns the result of processing a mixed context argument.
   *
   * The context value is processed into a string variable according to the
   *  following flowchart:
   *  - NULL or empty: No context will be prepended to the StateBase key name.
   *  - A string: The string is used as the variable prefix. Characters are
   *    replaced per the result of ::replacementCharacters().
   *  - An array: Each item in the array is individually processed, and the
   *    resultant array is concatenated into a dot-delimited string to use as
   *    the variable prefix.
   *  - A callable: The callable's return value will be processed and used as
   *    the variable prefix. A reference to the current instance of this object
   *    is provided as the sole callback argument.
   *  - An object that contains a ::toString() public method: The processed
   *    result of the ::toString() method is processed for return.
   *  - Any other value is cast to string and processed for return.
   *
   * @param mixed $context
   *  The argument to process.
   *
   * @return string|null
   *  The result of the stringification and concatenation process.
   */
  protected function processContext(mixed $context): ?string {
    // Return NULL for any empty values.
    if (empty($context)) {
      return NULL;
    }
    if (is_string($context)) {
      return str_replace(
        array_keys(static::replacementCharacters()),
        array_values(static::replacementCharacters()),
        $context
      );
    }
    if (is_array($context)) {
      return implode('.', array_map(
        fn ($item) => $this->processContext($item),
        $context
      ));
    }
    if (is_callable($context)) {
      return $this->processContext( $context($this) );
    }
    if (is_object($context) && method_exists($context, 'toString')) {
      return $this->processContext( $context->toString() );
    }
    return $this->processContext( (string)$context) ;
  }

  /**
   * @inheritDoc
   */
  abstract public function name(): string;

  /**
   *  An optional value that provides context for the access of StateBase
   *  key/value store values. Override this method to provide a context per
   *  ::processContext().
   *
   * @return mixed
   */
  protected function context(): mixed {
    return NULL;
  }

  /**
   * Returns an associative array of replacement string values, each keyed by
   * their respective match characters.
   *
   * @return string[]
   */
  protected function replacementCharacters(): array {
    return [
      '#' => '',
      '$' => '',
    ];
  }

  /**
   * Returns the concatenated context (if any) and variable name.
   *
   * @return string
   */
  protected function key(): string {
    return implode(':', array_filter([
      $this->processContext( $this->context() ),
      $this->name(),
    ]));
  }


  /**
   * @inheritDoc
   */
  public function setState(\Drupal\Core\State\StateInterface $state): static {
    $this->state = $state;
    return $this;
  }

  /**
   * Returns the assigned state service.
   *
   * @return \Drupal\Core\State\StateInterface
   */
  protected function state(): \Drupal\Core\State\StateInterface {
    return $this->state;
  }

  /**
   * @inheritDoc
   */
  public function set($value): static {
    $this->state()->set($this->key(), $value);
    return $this;
  }

  /**
   * @inheritDoc
   */
  public function get($default = NULL): mixed {
    return $this->state()->get($this->key(), $default);
  }

  /**
   * @inheritDoc
   */
  public function setArrayValue(string|int $key, mixed $value): static {
    $state_value = static::asArray( $this->get() );
    $state_value[$key] = $value;
    $this->set($state_value);
    return $this;
  }

  /**
   * @inheritDoc
   */
  public function arrayValue(string|int $key): mixed {
    return $this->get()[$key] ?? NULL;
  }

  /**
   * @inheritDoc
   */
  public function hasArrayValue($key): bool {
    $value = $this->get();
    return is_array($value) && isset( $value[$key] );
  }

  /**
   * @inheritDoc
   */
  public function delete(): static {
    $this->state()->delete( $this->key() );
    return $this;
  }

}
