<?php

declare(strict_types=1);

namespace Drupal\Tests\dx_toolkit\Unit;

use Drupal\dx_toolkit\ArrayUtilities;
use Drupal\Tests\UnitTestCase;

/**
 * Tests the ArrayUtilities class.
 *
 * @coversDefaultClass \Drupal\dx_toolkit\ArrayUtilities
 * @group dx_toolkit
 */
class ArrayUtilitiesTest extends UnitTestCase {

  /**
   * Tests arrayKeysCombined with single array.
   *
   * @covers ::arrayKeysCombined
   */
  public function testArrayKeysCombinedSingleArray(): void {
    $input = ['a' => 1, 'b' => 2, 'c' => 3];
    $result = ArrayUtilities::arrayKeysCombined($input);
    $expected = ['a' => 'a', 'b' => 'b', 'c' => 'c'];
    $this->assertEquals($expected, $result);
  }

  /**
   * Tests arrayKeysCombined with two arrays.
   *
   * @covers ::arrayKeysCombined
   */
  public function testArrayKeysCombinedTwoArrays(): void {
    $array1 = ['a' => 1, 'b' => 2, 'c' => 3];
    $array2 = ['x' => 10, 'y' => 20, 'z' => 30];
    $result = ArrayUtilities::arrayKeysCombined($array1, $array2);
    $expected = ['a' => 'x', 'b' => 'y', 'c' => 'z'];
    $this->assertEquals($expected, $result);
  }

  /**
   * Tests arrayMapMerged flattens array_map results.
   *
   * @covers ::arrayMapMerged
   */
  public function testArrayMapMerged(): void {
    $utilities = new class extends ArrayUtilities {};
    $callback = fn($item) => [$item * 2, $item * 3];
    $result = $utilities->arrayMapMerged($callback, [1, 2, 3]);
    $expected = [[2, 3], [4, 6], [6, 9]];
    $this->assertEquals($expected, $result);
  }

  /**
   * Tests arrayKeyColumn extracts column while preserving keys.
   *
   * @covers ::arrayKeyColumn
   */
  public function testArrayKeyColumn(): void {
    $utilities = new class extends ArrayUtilities {};
    $input = [
      'first' => ['name' => 'Alice', 'age' => 30],
      'second' => ['name' => 'Bob', 'age' => 25],
      'third' => ['name' => 'Charlie', 'age' => 35],
    ];
    $result = $utilities->arrayKeyColumn($input, 'name');
    $expected = [
      'first' => 'Alice',
      'second' => 'Bob',
      'third' => 'Charlie',
    ];
    $this->assertEquals($expected, $result);
  }

  /**
   * Tests arraySortByKeys sorts target by source keys.
   *
   * @covers ::arraySortByKeys
   */
  public function testArraySortByKeys(): void {
    $utilities = new class extends ArrayUtilities {};
    $target = ['c' => 3, 'a' => 1, 'b' => 2];
    $keySource = ['a' => 0, 'b' => 0, 'c' => 0];
    $result = $utilities->arraySortByKeys($target, $keySource);
    $expected = ['a' => 1, 'b' => 2, 'c' => 3];
    $this->assertEquals($expected, $result);
  }

  /**
   * Tests arraySortByKeys with partial key match.
   *
   * @covers ::arraySortByKeys
   */
  public function testArraySortByKeysPartialMatch(): void {
    $utilities = new class extends ArrayUtilities {};
    $target = ['c' => 3, 'a' => 1, 'b' => 2, 'd' => 4];
    $keySource = ['a' => 0, 'c' => 0];
    $result = $utilities->arraySortByKeys($target, $keySource);
    $expected = ['a' => 1, 'c' => 3];
    $this->assertEquals($expected, $result);
  }

}
