<?php

declare(strict_types=1);

namespace Drupal\Tests\dx_toolkit\Unit;

use Drupal\dx_toolkit\Color;
use Drupal\Tests\UnitTestCase;

/**
 * Tests the Color utility class.
 *
 * @coversDefaultClass \Drupal\dx_toolkit\Color
 * @group dx_toolkit
 */
class ColorTest extends UnitTestCase {

  /**
   * Tests hexToRgba conversion.
   *
   * @covers ::hexToRgba
   */
  public function testHexToRgba(): void {
    $result = Color::hexToRgba('#abc');
    $this->assertEquals('rgba(170, 187, 204, 1)', $result);

    $result = Color::hexToRgba('#aabbcc', 0.5);
    $this->assertEquals('rgba(170, 187, 204, 0.5)', $result);
  }

  /**
   * Tests contrast50 calculation.
   *
   * @covers ::contrast50
   */
  public function testContrast50(): void {
    $this->assertEquals('#ffffff', Color::contrast50('#000000'));
    $this->assertEquals('#000000', Color::contrast50('#ffffff'));
  }

  /**
   * Tests contrastYIQ calculation.
   *
   * @covers ::contrastYIQ
   */
  public function testContrastYIQ(): void {
    $this->assertEquals('#ffffff', Color::contrastYIQ('#000000'));
    $this->assertEquals('#000000', Color::contrastYIQ('#ffffff'));
  }

  /**
   * Tests luminance calculation.
   *
   * @covers ::luminance
   */
  public function testLuminance(): void {
    $black = Color::luminance('#000000');
    $this->assertEqualsWithDelta(0.0, $black, 0.01);

    $white = Color::luminance('#ffffff');
    $this->assertEqualsWithDelta(1.0, $white, 0.01);
  }

  /**
   * Tests luminance difference calculation.
   *
   * @covers ::luminanceDifference
   */
  public function testLuminanceDifference(): void {
    $diff = Color::luminanceDifference('#000000', '#ffffff');
    $this->assertEqualsWithDelta(21.0, $diff, 0.01);
  }

  /**
   * Tests contrastLuminance calculation.
   *
   * @covers ::contrastLuminance
   */
  public function testContrastLuminance(): void {
    $this->assertEquals('#ffffff', Color::contrastLuminance('#336699'));
  }

  /**
   * Tests calculateBestContrast.
   *
   * @covers ::calculateBestContrast
   */
  public function testCalculateBestContrast(): void {
    $result = Color::calculateBestContrast(
      '#336699',
      ['#000000', '#ffffff']
    );
    $this->assertEquals('#ffffff', $result);
  }

  /**
   * Tests calculateBestContrast with empty array.
   *
   * @covers ::calculateBestContrast
   */
  public function testCalculateBestContrastEmpty(): void {
    $result = Color::calculateBestContrast('#336699', []);
    $this->assertNull($result);
  }

}
