<?php

declare(strict_types=1);

namespace Drupal\Tests\dx_toolkit\Unit;

use Drupal\dx_toolkit\Json;
use Drupal\Tests\UnitTestCase;

/**
 * Tests the Json utility class.
 *
 * @coversDefaultClass \Drupal\dx_toolkit\Json
 * @group dx_toolkit
 */
class JsonTest extends UnitTestCase {

  /**
   * Tests JSON string cleaning removes control characters.
   *
   * @covers ::cleanJsonString
   */
  public function testCleanJsonStringRemovesControlCharacters(): void {
    $json = '{"name":' . chr(0) . '"test"' . chr(31) . '}';
    $cleaned = Json::cleanJsonString($json);
    $this->assertEquals('{"name":"test"}', $cleaned);
  }

  /**
   * Tests JSON string cleaning removes BOM.
   *
   * @covers ::cleanJsonString
   */
  public function testCleanJsonStringRemovesBom(): void {
    $json = hex2bin('efbbbf') . '{"name":"test"}';
    $cleaned = Json::cleanJsonString($json);
    $this->assertEquals('{"name":"test"}', $cleaned);
  }

  /**
   * Tests JSON string cleaning handles valid JSON.
   *
   * @covers ::cleanJsonString
   */
  public function testCleanJsonStringPreservesValidJson(): void {
    $json = '{"name":"test","value":123}';
    $cleaned = Json::cleanJsonString($json);
    $this->assertEquals($json, $cleaned);
  }

  /**
   * Tests decode with cleaning enabled.
   *
   * @covers ::decode
   */
  public function testDecodeWithCleaning(): void {
    $json = '{"name":' . chr(0) . '"test"}';
    $result = Json::decode($json, TRUE);
    $this->assertEquals(['name' => 'test'], $result);
  }

  /**
   * Tests decode without cleaning.
   *
   * @covers ::decode
   */
  public function testDecodeWithoutCleaning(): void {
    $json = '{"name":"test"}';
    $result = Json::decode($json, FALSE);
    $this->assertEquals(['name' => 'test'], $result);
  }

  /**
   * Tests decode with BOM-prefixed JSON.
   *
   * @covers ::decode
   */
  public function testDecodeWithBom(): void {
    $json = hex2bin('efbbbf') . '{"name":"test"}';
    $result = Json::decode($json, TRUE);
    $this->assertEquals(['name' => 'test'], $result);
  }

}
