# DX Toolkit Demo Module

This module demonstrates the utility and benefits of the **ServiceInjector** plugin system.

## What This Demonstrates

1. **Custom ServiceInjector Plugins**
   - ViewStorage plugin (simple, no deriver)
   - ConfigFactory plugin (with deriver, moved from core module)
2. **Using Generated Services** - ContentAuditor service
3. **Cleaner Dependency Injection** - Compare traditional vs ServiceInjector
4. **Practical Example** - Site content auditing

## Installation

```bash
lando drush en dx_toolkit_demo -y
```

## Usage

### Drush Commands

**Site-wide audit:**
```bash
lando drush dx:demo:audit
```

**Audit specific entity type:**
```bash
lando drush dx:demo:audit-type node
lando drush dx:demo:audit-type taxonomy_term
lando drush dx:demo:audit-type user
```

## The ServiceInjector Advantage

### Traditional Approach (VERBOSE)

```php
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;

class ContentAuditor {

  protected EntityStorageInterface $nodeStorage;
  protected EntityStorageInterface $termStorage;
  protected EntityStorageInterface $userStorage;
  protected Config $siteConfig;

  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    ConfigFactoryInterface $configFactory,
  ) {
    // Verbose initialization in constructor body
    $this->nodeStorage = $entityTypeManager->getStorage('node');
    $this->termStorage = $entityTypeManager->getStorage('taxonomy_term');
    $this->userStorage = $entityTypeManager->getStorage('user');
    $this->siteConfig = $configFactory->getEditable('system.site');
  }
}
```

**services.yml:**
```yaml
services:
  my_module.content_auditor:
    class: Drupal\my_module\ContentAuditor
    arguments:
      - '@entity_type.manager'
      - '@config.factory'
```

**Problems:**
- ❌ Verbose constructor body
- ❌ No type hints for storage handlers
- ❌ Manual factory method calls
- ❌ Easy to forget to initialize a storage
- ❌ Hard to test (must mock entire EntityTypeManager)

### ServiceInjector Approach (CLEAN)

```php
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Config\Config;

class ContentAuditor {

  public function __construct(
    protected EntityStorageInterface $nodeStorage,
    protected EntityStorageInterface $termStorage,
    protected EntityStorageInterface $userStorage,
    protected Config $siteConfig,
  ) {}

  // No initialization code needed!
}
```

**services.yml:**
```yaml
services:
  my_module.content_auditor:
    class: Drupal\my_module\ContentAuditor
    arguments:
      - '@service_injector.node.storage'
      - '@service_injector.taxonomy_term.storage'
      - '@service_injector.user.storage'
      - '@service_injector.system.site.config'
```

**Benefits:**
- ✅ Clean, minimal constructor
- ✅ Full type hints
- ✅ Dependencies clearly declared
- ✅ Easy to test (mock individual storage handlers)
- ✅ Services ready to use immediately
- ✅ Self-documenting code

## How It Works

### 1. ServiceInjector Plugins Generate Services

The `EntityStorage` plugin (in dx_toolkit) automatically creates services for all entity types:
- `service_injector.node.storage`
- `service_injector.taxonomy_term.storage`
- `service_injector.user.storage`
- ... etc (65+ services)

The `ConfigFactory` plugin (in dx_toolkit) creates config services:
- `service_injector.system.site.config`
- `service_injector.node.settings.config`
- `service_injector.user.settings.config`
- ... etc

### 2. Custom Plugin Example

This demo module includes a custom `ViewStorage` plugin:

```php
/**
 * @ServiceInjector(
 *   id = "view_storage",
 *   factoryService = "@entity_type.manager",
 *   factoryMethod = "getStorage",
 *   factoryArguments = ["view"],
 *   serviceClass = "Drupal\Core\Entity\EntityStorageInterface",
 *   servicePrefix = "service_injector",
 *   serviceSuffix = "storage"
 * )
 */
class ViewStorage extends ServiceInjectorBase {}
```

This generates: `service_injector.view.storage`

No deriver needed for single services!

### 3. Using Generated Services

Simply reference them in `services.yml`:

```yaml
your_service:
  arguments:
    - '@service_injector.node.storage'
    - '@service_injector.system.site.config'
```

## Testing the Demo

```bash
# Enable module
lando drush en dx_toolkit_demo -y

# Run site audit
lando drush dx:demo:audit

# Audit nodes
lando drush dx:demo:audit-type node
```

## Real-World Use Cases

ServiceInjector is ideal when you need:

1. **Multiple Storage Handlers** - Forms, services, commands that work with multiple entity types
2. **Config Objects** - Services that need multiple config objects
3. **Type Safety** - Proper type hints for better IDE support and static analysis
4. **Testability** - Easier to mock specific dependencies
5. **Readability** - Cleaner, more maintainable code

## Creating Your Own ServiceInjector Plugins

See `SERVICEINJECTOR_EXAMPLE.md` in the parent dx_toolkit module for detailed examples.

Quick template:

```php
/**
 * @ServiceInjector(
 *   id = "my_plugin",
 *   factoryService = "@my.factory",
 *   factoryMethod = "create",
 *   serviceClass = "My\Interface",
 *   deriver = "My\Deriver"  // Optional
 * )
 */
class MyPlugin extends ServiceInjectorBase {}
```

## Performance

ServiceInjector adds **zero runtime overhead**:
- Services generated at compile-time
- No dynamic lookups or lazy loading
- Same performance as manually defined services
- Slightly slower container builds (negligible)

## Summary

ServiceInjector transforms the factory pattern from verbose boilerplate into clean, declarative dependency injection. It's especially powerful for:

- Services needing multiple entity storage handlers
- Services needing multiple config objects
- Reducing constructor complexity
- Improving code readability
- Making tests easier to write

Try it yourself - compare the ContentAuditor service in this demo to what you'd write traditionally!
