<?php

declare(strict_types=1);

namespace Drupal\dx_toolkit_demo\Commands;

use Drupal\dx_toolkit_demo\ContentAuditorInterface;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;

/**
 * Drush commands demonstrating ServiceInjector utility.
 */
final class DemoCommands extends DrushCommands
{

  /**
   * Constructs a DemoCommands object.
   *
   * @param \Drupal\dx_toolkit_demo\ContentAuditorInterface $auditor
   *   The content auditor service.
   */
  public function __construct(
    protected ContentAuditorInterface $auditor,
  ) {
    parent::__construct();
  }

  /**
   * Displays site-wide content audit.
   */
  #[CLI\Command(name: 'dx:demo:audit')]
  #[CLI\Usage(
    name: 'dx:demo:audit',
    description: 'Shows site content statistics'
  )]
  public function audit(): void {
    $stats = $this->auditor->auditSite();

    $this->io()->title('ServiceInjector Demo: Site Content Audit');

    $this->io()->table(
      ['Metric', 'Value'],
      [
        ['Site Name', $stats['site_name']],
        ['Nodes', $stats['node_count']],
        ['Taxonomy Terms', $stats['term_count']],
        ['Users', $stats['user_count']],
        ['Total Content Items', $stats['total_content']],
      ]
    );

    $this->io()->success('Audit complete!');
  }

  /**
   * Displays detailed audit for specific entity type.
   *
   * @param string $entity_type
   *   Entity type ID (node, taxonomy_term, or user).
   */
  #[CLI\Command(name: 'dx:demo:audit-type')]
  #[CLI\Argument(
    name: 'entity_type',
    description: 'Entity type to audit (node, taxonomy_term, user)'
  )]
  #[CLI\Usage(
    name: 'dx:demo:audit-type node',
    description: 'Shows detailed node audit'
  )]
  public function auditType(string $entity_type): void {
    try {
      $data = $this->auditor->auditByType($entity_type);

      $this->io()->title(
        "ServiceInjector Demo: {$entity_type} Audit"
      );
      $this->io()->text("Total: {$data['total_count']}");

      if (!empty($data['entities'])) {
        $this->io()->table(
          ['ID', 'Label', 'Bundle'],
          array_map(
            fn ($entity) => [
              $entity['id'],
              $entity['label'],
              $entity['bundle'],
            ],
            $data['entities']
          )
        );
      }

      $this->io()->success('Audit complete!');
    }
    catch (\InvalidArgumentException $e) {
      $this->io()->error($e->getMessage());
    }
  }

}
