<?php

declare(strict_types=1);

namespace Drupal\dx_toolkit;

/**
 * Class Utilities
 *  A collection of array utility functions.
 */
abstract class ArrayUtilities
{
  /**
   * Combines the keys of one or two arrays into a single array. The first
   * array's keys are used as the keys of the returned array, and the second
   * array's keys are used as the values of the returned array. If the second
   * array is not provided, the first array's keys are used as both the keys
   * and values of the returned array.
   *
   * @param array $array1
   *  The first array.
   *
   * @param array|null $array2
   *  Optional. The second array.
   *
   * @return array
   */
  public static function arrayKeysCombined(
    array $array1,
    array $array2 = NULL
  ): array {
    $keys1 = array_keys($array1);
    $keys2 = $array2 ? array_keys($array2) : $keys1;
    return array_combine($keys1, $keys2);
  }

  /**
   * Used to flatten the results of an array_map invocation that expects to
   * return an array of arrays. The provided array will have the callback mapped
   * to each of its items, then the resultant array of arrays is flattened and
   * returned.
   *
   * Note that the $array value should be keyed ordinally. Use array_values()
   * to reindex an associative array before passing it to this function.
   * Example:
   * ```
   * $results = array_map_merged(
   *  fn ($item) => some_function_returning_array($item),
   *  array_value($associative_array),
   * );
   *
   * @param array $array
   *  The nested array against whose values the provided array_map function
   *  should be invoked.
   *
   * @param callable $callback
   *  The callback to invoke against each value in the provided array.
   *
   * @return array
   *  An array containing the flattened results of all the array map
   *  invocations.
   */
  public function arrayMapMerged(
    callable $callback,
    array $array,
    array ...$arrays
  ): array {
    return [...array_map($callback, $array, ...$arrays)];
  }

  /**
   * Returns an array column's values while maintaining original key
   * associations.
   *
   * While the array_column() function will key returned values ordinally, or by
   * an optionally specified key column, this function will key the returned
   * values each by the parent array's respective key.
   *
   * @param array $array
   *  The array from which to extract the column.
   *
   * @param string $column_name
   *  The name of the column to extract.
   *
   * @return array
   */
  public function arrayKeyColumn(array $array, string $column_name): array {
    return array_map(fn (array $item) => $item[$column_name], $array);
  }

  /**
   * Returns a new array derived from the original target array, in which the
   * items have been sorted by the keys of the source array.
   *
   * @param array $target
   *  The array to sort.
   *
   * @param array $key_source
   *  The array whose keys will be used to sort the target array.
   *
   * @param bool $by_reference
   *  Whether to sort the target array by reference.
   *
   * @return array
   *  The sorted array.
   */
  public function arraySortByKeys(
    array $target,
    array $key_source,
    bool $by_reference = False
  ): array {
    $sorted_target = [];
    foreach (array_keys($key_source) as $key) {
      if (array_key_exists($key, $target)) {
        $sorted_target[$key] = $target[$key];
      }
    }
    if ($by_reference) {
      $target = &$sorted_target;
    }
    return $sorted_target;
  }

}
