<?php

declare(strict_types=1);

namespace Drupal\dx_toolkit;

use Drupal\Core\Extension\Extension;
use Drupal\Core\Extension\ModuleHandlerInterface;

/**
 * Trait ExtensionTrait
 *  Provides common implementations that fulfill most of the requirements of
 *  \Drupal\dx_toolkit\ExtensionInterface.
 *
 * NOTE: This trait uses \Drupal::moduleHandler() as a service locator
 * pattern to support static method access. This is an acceptable exception
 * to the no-container rule (R006) because:
 * 1. The trait provides static utility methods (getModule, getExtensionPath)
 * 2. Static methods cannot access injected dependencies
 * 3. Alternative would require all calling code to inject dependencies
 *
 * For class implementations that need ModuleHandlerInterface in instance
 * methods, inject the service directly instead of using this trait's
 * getModuleHandler() method:
 *
 * @code
 * public function __construct(
 *   protected ModuleHandlerInterface $moduleHandler,
 * ) {}
 * @endcode
 */
trait ExtensionTrait
{

  /**
   * Returns the name of the currently implementing module.
   *
   * @return string
   */
  abstract public static function name(): string;

  /**
   * Returns the label to display for this module.
   *
   * @return string
   */
  abstract public static function label(): string;

  /**
   * Returns a reference to the Module Handler service.
   *
   * This method uses the service locator pattern to support static
   * method access. For instance methods, inject ModuleHandlerInterface
   * directly via constructor instead.
   *
   * @return ModuleHandlerInterface
   */
  protected static function moduleHandler(): ModuleHandlerInterface {
    return \Drupal::moduleHandler();
  }

  /**
   * Returns a reference to the Core Extension object that represents this
   * module.
   *
   * @return Extension
   */
  public static function module(): Extension {
    return static::moduleHandler()->getModule( static::name() );
  }

  /**
   * Return this extension's relative path.
   *
   * The extension's relative path is the path fragment that acts as the
   * relative file path, and is typically the path following the web root.
   *
   * @see ::root()
   *
   * @return string
   */
  public static function extensionPath(): string {
    return static::module()->getPath();
  }

  /**
   * Returns the root path to the current Drupal instance.
   *
   * The root path is typically the path fragment found at the beginning of an
   * absolute file path, up to the relative path fragment. Typically, this will
   * be the web root.
   *
   * @see ::extensionPath().
   *
   * @return string
   */
  public static function root(): string {
    return DRUPAL_ROOT;
  }

}
