<?php

declare(strict_types=1);

namespace Drupal\dx_toolkit;

/**
 * Class Json
 *  Extends the core JSON utility class in order to provide sanitization for
 *  JSON strings.
 *
 * @package Drupal\dx_toolkit
 */
class Json extends \Drupal\Component\Serialization\Json {

  /**
   * @inheritDoc
   */
  public static function decode($string, bool $clean_string = TRUE) {
    if ($clean_string) {
      $string = static::cleanJsonString($string);
    }
    return parent::decode($string);
  }

  /**
   * Cleans unprintable and any BOM (byte-order marks) from a JSON string.
   *
   * @param string $json
   *
   * @return string
   */
  public static function cleanJsonString(string $json): string {
    // @see https://stackoverflow.com/a/20845642/6679439
    // Remove unprintable characters (0 - 31, 127).
    $chars = [
      ...range(0, 31),
      127
    ];
    $json = array_reduce(
      $chars,
      fn($str, $char) => str_replace(chr($char), "", $str),
      $json
    );
    // Remove any BOM.
    if (str_starts_with( bin2hex($json), static::bomHex())) {
      $json = substr($json, 3);
    }
    return $json;
  }

  /**
   * Returns the BOM (byte-order mark) in hex format.
   *
   * The hex code 'efbbbf' represents the UTF-8 Byte Order Mark (BOM).
   * When text is encoded in UTF-8, some editors or systems add these three
   * bytes at the beginning of a file to indicate UTF-8 encoding:
   * EF = 239 in decimal
   * BB = 187 in decimal
   * BF = 191 in decimal
   *
   * The BOM can cause issues when parsing JSON because it's invisible but
   * still present in the string.
   *
   * @return string
   */
  protected static function bomHex(): string {
    return 'efbbbf';
  }

}
