<?php

declare(strict_types=1);

namespace Drupal\dx_toolkit\Plugin;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Config\FileStorage;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\dx_toolkit\Extension;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class EntityGeneratorBase
 *  Provides a base class for implementing EntityGenerator plugin types.
 *
 * @package Drupal\dx_toolkit\Plugin
 */
abstract class EntityGeneratorBase extends PluginBase
  implements EntityGeneratorInterface, ContainerFactoryPluginInterface
{
  use StringTranslationTrait;

  /**
   * Entity storage handler for the source entity type.
   *
   * @var EntityStorageInterface
   */
  protected EntityStorageInterface $sourceEntityStorage;

  /**
   * Entity storage handler for the target entity type.
   *
   * @var EntityStorageInterface
   */
  protected EntityStorageInterface $targetEntityStorage;

  /**
   * Entity definition for the source entity type.
   *
   * @var EntityTypeInterface
   */
  protected EntityTypeInterface $sourceEntityDefinition;

  /**
   * Entity definition for the target entity type.
   *
   * @var EntityTypeInterface
   */
  protected EntityTypeInterface $targetEntityDefinition;

  /**
   * Bundle info for the source entity type.
   *
   * @var array
   */
  protected array $sourceBundleInfo;

  /**
   * The path for this plugin's source module.
   *
   * @var string
   */
  protected string $extensionPath;

  /**
   * @param array $configuration
   * @param $plugin_id
   * @param $plugin_definition
   * @param EntityTypeManagerInterface $entity_type_manager
   * @param EntityTypeBundleInfoInterface $entity_bundle_info
   * @param ModuleHandlerInterface $module_handler
   *
   * @throws InvalidPluginDefinitionException
   * @throws PluginNotFoundException
   */
  public function __construct(
    $configuration,
    $plugin_id,
    $plugin_definition,
    EntityTypeManagerInterface $entity_type_manager,
    protected EntityTypeBundleInfoInterface $entity_bundle_info,
    protected ModuleHandlerInterface $module_handler,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->sourceEntityStorage = $entity_type_manager
      ->getStorage($this->getSourceEntityTypeId());
    $this->targetEntityStorage = $entity_type_manager
      ->getStorage($this->getTargetEntityTypeId());
    $this->sourceEntityDefinition = $entity_type_manager
      ->getDefinition($this->getSourceEntityTypeId());
    $this->targetEntityDefinition = $entity_type_manager
      ->getDefinition($this->getTargetEntityTypeId());
    $this->sourceBundleInfo = $entity_bundle_info
      ->getBundleInfo($this->getSourceEntityTypeId());
    $this->extensionPath = $module_handler
      ->getModule($this->sourceModule())->getPath();
  }

  /**
   * @inheritDoc
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('module_handler')
    );
  }

  /**
   * @inheritDoc
   */
  public function getConfigName(): string {
    return $this->pluginDefinition['configName'];
  }

  /**
   * @inheritDoc
   */
  public function getConfigPath(): string {
    return $this->pluginDefinition['configPath'];
  }

  /**
   * @inheritDoc
   */
  public function sourceModule(): string {
    return $this->pluginDefinition['sourceModule'];
  }

  /**
   * @inheritDoc
   */
  public function getTargetEntityTypeId(): string {
    return $this->pluginDefinition['targetEntityTypeId'];
  }

  /**
   * @inheritDoc
   */
  public function getSourceEntityTypeId(): string {
    return $this->pluginDefinition['sourceEntityTypeId'];
  }

  /**
   * @inheritDoc
   */
  public function bundleName(): string {
    return $this->pluginDefinition['bundleName'];
  }

  /**
   * @inheritDoc
   */
  public function getBaseConfigurationData(): array {
    $path = implode(DIRECTORY_SEPARATOR, [
      Extension::root(),
      $this->extensionPath,
      $this->getConfigPath(),
    ]);
    $file_name = $this->getConfigName();
    $file_storage = new FileStorage($path);
    return ($file_storage->exists($file_name))
      ? $file_storage->read($file_name)
      : [];
  }

  /**
   * @inheritDoc
   */
  public function getTargetEntityConfiguration(): array {
    return NestedArray::mergeDeep(
      $this->getBaseConfigurationData(),
      $this->customConfiguration()
    );
  }

  /**
   * @inheritDoc
   */
  public function generateEntity(): static {
    if (!$this->hasTargetEntity()) {
      $entity = $this->targetEntityStorage
        ->create($this->getTargetEntityConfiguration());
      $entity->save();
    }
    return $this;
  }

  /**
   * @inheritDoc
   */
  public function deleteTargetEntity(): static {
    if ($this->hasTargetEntity()) {
      $this->targetEntityStorage->delete([$this->getTargetEntity()]);
    }
    return $this;
  }

  /**
   * @inheritDoc
   */
  public function getTargetEntity(): ?EntityInterface {
    return $this->targetEntityStorage->load($this->targetId());
  }

  /**
   * @inheritDoc
   */
  public function hasTargetEntity(): bool {
    return !empty($this->getTargetEntity());
  }

  /**
   * Returns the label for this plugin's source bundle.
   *
   * @return string
   */
  protected function getSourceBundleLabel(): string {
    return $this?->sourceBundleInfo[$this->bundleName()]['label'] ?? '';
  }

}
