<?php

declare(strict_types=1);

namespace Drupal\dx_toolkit;

/**
 * Interface StateInterface
 *  Defines an object that provides an OO wrapper around individual key/value
 *  pairs provided by the StateBase key/value store.
 *
 * @package Drupal\dx_toolkit
 */
interface StateInterface
{

  /**
   * Assigns the service to use as the StateBase
   *
   * @param \Drupal\Core\State\StateInterface $state
   *
   * @return $this
   */
  public function setState(\Drupal\Core\State\StateInterface $state): static;

  /**
   * Returns the variable name to use for accessing the StateBase key/value
   * store.
   *
   * @return string
   */
  public function name(): string;

  /**
   * Sets the state value.
   *
   * @param $value
   *
   * @return $this
   *  Reference to self.
   */
  public function set($value): static;

  /**
   * Returns the state's value. Returns the provided default value if no state
   * value exists.
   *
   * @param null $default
   *
   * @return mixed|NULL
   */
  public function get($default = NULL): mixed;

  /**
   * Sets a value to the key specified in the state value. In the event that
   * the state's value is not already an array, it will be overwritten with
   * a single-element array containing the provided state value.
   *
   * @param int|string $key
   * The key of the state value's array.
   *
   * @param mixed $value
   * The new state value.
   *
   * @return $this
   *  Reference to self.
   */
  public function setArrayValue(int|string $key, mixed $value): static;

  /**
   * Returns the value state's assigned array. Returns NULL in the event that
   * either the current state value is not an array, or the array key is
   * invalid.
   *
   * @param string|int $key
   *  The key of the state value's array element to return.
   *
   * @return mixed
   *  The state's array element value.
   */
  public function arrayValue(string|int $key): mixed;

  /**
   * Returns TRUE if the state value is an array and contains a value in the
   * element specified by the key argument.
   *
   * @param int|string $key
   *  The array key to check for existence.
   *
   * @return bool
   */
  public function hasArrayValue(int|string $key): bool;

  /**
   * Deletes the state value.
   *
   * @return $this
   *  Reference to self.
   */
  public function delete(): static;

}
