<?php

declare(strict_types=1);

namespace Drupal\Tests\dx_toolkit\Unit;

use Drupal\Core\State\StateInterface;
use Drupal\dx_toolkit\StateBase;
use Drupal\Tests\UnitTestCase;

/**
 * Tests for StateBase wrapper class.
 *
 * @group dx_toolkit
 * @coversDefaultClass \Drupal\dx_toolkit\StateBase
 */
class StateBaseTest extends UnitTestCase {

  /**
   * Mock state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $stateService;

  /**
   * Test state implementation.
   *
   * @var \Drupal\dx_toolkit\StateBase
   */
  protected $testState;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->stateService = $this->createMock(StateInterface::class);

    $this->testState = new class extends StateBase {

      /**
       * {@inheritdoc}
       */
      public function name(): string {
        return 'test_state';
      }

      /**
       * {@inheritdoc}
       */
      protected function context(): string {
        return 'test';
      }

      /**
       * Expose state() for testing.
       */
      public function getState(): StateInterface {
        return $this->state();
      }

    };
  }

  /**
   * Tests setState() fluent interface.
   *
   * @covers ::setState
   */
  public function testSetStateFluentInterface(): void {
    $result = $this->testState->setState($this->stateService);
    $this->assertSame($this->testState, $result);
  }

  /**
   * Tests that state() returns service after setState().
   *
   * @covers ::state
   * @covers ::setState
   */
  public function testStateReturnsServiceAfterSetState(): void {
    $this->testState->setState($this->stateService);
    $this->assertSame($this->stateService, $this->testState->getState());
  }

  /**
   * Tests set() method integration.
   *
   * @covers ::set
   * @covers ::key
   */
  public function testSet(): void {
    $this->stateService
      ->expects($this->once())
      ->method('set')
      ->with('test:test_state', 'test_value');

    $result = $this->testState->setState($this->stateService)->set('test_value');
    $this->assertSame($this->testState, $result);
  }

  /**
   * Tests get() method integration.
   *
   * @covers ::get
   * @covers ::key
   */
  public function testGet(): void {
    $this->stateService
      ->expects($this->once())
      ->method('get')
      ->with('test:test_state', NULL)
      ->willReturn('stored_value');

    $result = $this->testState->setState($this->stateService)->get();
    $this->assertSame('stored_value', $result);
  }

  /**
   * Tests get() with default value.
   *
   * @covers ::get
   */
  public function testGetWithDefault(): void {
    $this->stateService
      ->expects($this->once())
      ->method('get')
      ->with('test:test_state', 'default')
      ->willReturn('default');

    $result = $this->testState->setState($this->stateService)->get('default');
    $this->assertSame('default', $result);
  }

  /**
   * Tests delete() method integration.
   *
   * @covers ::delete
   * @covers ::key
   */
  public function testDelete(): void {
    $this->stateService
      ->expects($this->once())
      ->method('delete')
      ->with('test:test_state');

    $result = $this->testState->setState($this->stateService)->delete();
    $this->assertSame($this->testState, $result);
  }

  /**
   * Tests setArrayValue() method.
   *
   * @covers ::setArrayValue
   */
  public function testSetArrayValue(): void {
    $existing = ['key1' => 'value1'];
    $expected = ['key1' => 'value1', 'key2' => 'value2'];

    $this->stateService
      ->expects($this->once())
      ->method('get')
      ->with('test:test_state', NULL)
      ->willReturn($existing);

    $this->stateService
      ->expects($this->once())
      ->method('set')
      ->with('test:test_state', $expected);

    $result = $this->testState
      ->setState($this->stateService)
      ->setArrayValue('key2', 'value2');

    $this->assertSame($this->testState, $result);
  }

  /**
   * Tests arrayValue() method.
   *
   * @covers ::arrayValue
   */
  public function testArrayValue(): void {
    $data = ['key1' => 'value1', 'key2' => 'value2'];

    $this->stateService
      ->expects($this->once())
      ->method('get')
      ->with('test:test_state', NULL)
      ->willReturn($data);

    $result = $this->testState
      ->setState($this->stateService)
      ->arrayValue('key2');

    $this->assertSame('value2', $result);
  }

  /**
   * Tests arrayValue() with missing key.
   *
   * @covers ::arrayValue
   */
  public function testArrayValueMissingKey(): void {
    $data = ['key1' => 'value1'];

    $this->stateService
      ->expects($this->once())
      ->method('get')
      ->with('test:test_state', NULL)
      ->willReturn($data);

    $result = $this->testState
      ->setState($this->stateService)
      ->arrayValue('missing');

    $this->assertNull($result);
  }

  /**
   * Tests hasArrayValue() method.
   *
   * @covers ::hasArrayValue
   */
  public function testHasArrayValue(): void {
    $data = ['key1' => 'value1', 'key2' => NULL];

    $this->stateService
      ->method('get')
      ->with('test:test_state', NULL)
      ->willReturn($data);

    $this->testState->setState($this->stateService);

    $this->assertTrue($this->testState->hasArrayValue('key1'));
    $this->assertFalse($this->testState->hasArrayValue('key2'));
    $this->assertFalse($this->testState->hasArrayValue('missing'));
  }

  /**
   * Tests key() generation.
   *
   * @covers ::key
   * @covers ::processContext
   */
  public function testKeyGeneration(): void {
    $this->stateService
      ->expects($this->once())
      ->method('set')
      ->with('test:test_state', 'value');

    $this->testState->setState($this->stateService)->set('value');
  }

}
