/* jslint white:true, multivar, this, browser:true */

/**
 * @file Handles the image selection UI component for DXPR Builder Drupal integration.
 */

(function ($, Drupal, drupalSettings, window) {
  "use strict";

  // Dependencies from other files (ensure they are loaded before this script)
  const getUrlsFromInput =
    window.dxprBuilder && window.dxprBuilder.getUrlsFromInput;
  const dxpr_builder_get_images =
    window.dxprBuilder && window.dxprBuilder.dxpr_builder_get_images;
  const hideImageStyleControls =
    window.dxprBuilder && window.dxprBuilder.hideImageStyleControls;
  const createThumbailFromDefault =
    window.dxprBuilder && window.dxprBuilder.createThumbailFromDefault;
  const createEntityBrowserButton =
    window.dxprBuilder && window.dxprBuilder.createEntityBrowserButton;
  const createFileUploadElement =
    window.dxprBuilder && window.dxprBuilder.createFileUploadElement;
  const createFileUploadButton =
    window.dxprBuilder && window.dxprBuilder.createFileUploadButton;

  /**
   * Change handler for the image style select element
   *
   * @param {jQuery} selectElement The select element for image styles
   * @param {string} delimiter The delimiter used between URLs in the input
   */
  function imageStyleChangeHandler(selectElement, delimiter) {
    if (!getUrlsFromInput || !dxpr_builder_get_images) {
      console.error(
        "DXPR Builder: Missing dependencies for image style change.",
      );
      return;
    }
    // Find the selected option and act on it
    const imageStyle = selectElement.val();

    // Get the image input containing the URL of the image
    const imageInput = selectElement.siblings(".form-control:first");
    // If a delimiter has been provided, it means multiple images are allowed,
    // so each image needs the image style applied
    if (delimiter) {
      // Create an array of the currently entered images
      const currentImages = getUrlsFromInput(imageInput, delimiter);

      // Create an array to hold the images with the new image style URLs
      const newImages = [];
      // Loop through each of the current images, creating an array with the new image URLs
      currentImages.forEach((fileUrl) => {
        dxpr_builder_get_images(
          false,
          imageStyle,
          fileUrl,
          imageInput,
          delimiter,
          newImages,
        );
      });
    } else {
      const fileUrl = imageInput.val();
      dxpr_builder_get_images(false, imageStyle, fileUrl, imageInput);
    }
  }

  /**
   * Create the select element users will use to select an image style
   *
   * @param {jQuery} input The input used as a reference for inserting the select element into the DOM
   * @param {string} delimiter The delimiter used between URLs in the input
   */
  function createImageStyleInput(input, delimiter) {
    if (!hideImageStyleControls) {
      console.error(
        "DXPR Builder: Missing dependencies for image style input creation.",
      );
      return;
    }

    // Let label; // Unused variable

    const inputValue = input[0].value;
    let selectedStyle = null;
    if (inputValue) {
      let matches = inputValue.match(/styles\/([^/]+)\/(public|private)/);
      if (matches && matches[1]) {
        [, selectedStyle] = matches;
      }
      if (!selectedStyle) {
        matches = inputValue.match(/imageStyle=([^&,]*)/);
        if (matches && matches[1]) {
          [, selectedStyle] = matches;
        }
      }
    }

    // Create the select element using the template
    const imageStyleSelectElement = renderTemplate(
      "partials/forms/image-style-select",
      {
        image_styles: drupalSettings.dxprBuilder.imageStyles,
        selected_key: selectedStyle, // Pass the determined selected style to the template
      },
    );

    // Append the newly created elements to the page
    input[0].parentNode.insertBefore(imageStyleSelectElement, input[0]);

    const imageStyleSelect = $(imageStyleSelectElement);

    // Add change event handler for native select when Chosen is not available
    imageStyleSelectElement.addEventListener("change", () => {
      imageStyleChangeHandler(imageStyleSelect, delimiter);
    });

    hideImageStyleControls(input);
  }

  /**
   * This function is used to launch the code in this script, and is
   * called by external scripts.
   *
   * @param {HTMLElement} input The input into which URLs should be inserted. The URLs will then
   *   become images in the DOM when the dialog is saved
   * @param {string} delimiter The delimiter used between URLs in the input
   */
  function backend_images_select(input, delimiter) {
    if (
      !createThumbailFromDefault ||
      !createEntityBrowserButton ||
      !createFileUploadElement ||
      !createFileUploadButton
    ) {
      console.error(
        "DXPR Builder: Missing dependencies for backend_images_select.",
      );
      return;
    }

    let inputElement = $(input);
    inputElement[0].style.display = "block";

    // Detach the input, wrap it, then re-attach the wrapped structure
    const originalParent = inputElement[0].parentNode;
    const detachedInputElement = inputElement[0].cloneNode(true);
    originalParent.removeChild(inputElement[0]);

    const wrappedContent = renderTemplate(
      "partials/wrappers/ac-select-image-wrappers",
      {
        content: detachedInputElement.outerHTML,
      },
    );
    originalParent.insertBefore(wrappedContent, originalParent.firstChild);

    // Re-select inputElement from its new position in the DOM
    inputElement = wrappedContent.querySelector(
      (input.tagName ? input.tagName.toLowerCase() : "input") +
        (input.id ? `#${input.id}` : "") +
        (input.className
          ? `.${input.className.trim().replace(/\s+/g, ".")}`
          : ""),
    );

    inputElement = $(inputElement);
    if (
      drupalSettings.dxprBuilder.mediaBrowser &&
      drupalSettings.dxprBuilder.mediaBrowser.length > 0
    ) {
      createEntityBrowserButton(inputElement);
    } else {
      createFileUploadElement(inputElement, delimiter, "image");
      createFileUploadButton(inputElement, "image");
    }
    createImageStyleInput(inputElement, delimiter);
    createThumbailFromDefault(inputElement, delimiter);

    inputElement.change({ input: inputElement, delimiter }, (event) => {
      inputElement.siblings(".preview:first").empty();
      createThumbailFromDefault(inputElement, delimiter);
    });
  }

  // Expose the public API function
  window.dxprBuilder = window.dxprBuilder || {};
  window.dxprBuilder.backend_images_select = backend_images_select;

  // Expose internal functions needed by other files (if any)
  // Example: window.dxprBuilder.imageStyleChangeHandler = imageStyleChangeHandler;
  // Example: window.dxprBuilder.createImageStyleInput = createImageStyleInput;
})(jQuery, Drupal, drupalSettings, window);
