<?php

namespace Drupal\dxpr_theme_helper;

use Drupal\ai\AiProviderPluginManager;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Service for generating color palettes using AI.
 */
class AiPaletteGenerator {

  use StringTranslationTrait;

  /**
   * The AI provider plugin manager.
   *
   * @var \Drupal\ai\AiProviderPluginManager|null
   */
  protected ?AiProviderPluginManager $aiProvider;

  /**
   * The system prompt for palette generation.
   *
   * @var string
   */
  protected const SYSTEM_PROMPT = <<<'PROMPT'
You are a professional color palette designer. Generate a cohesive, visually appealing color palette based on the user's request.

Return ONLY valid JSON with these exact keys and hex color values (no markdown, no explanation):
{
  "base": "#XXXXXX",
  "basetext": "#XXXXXX",
  "link": "#XXXXXX",
  "accent1": "#XXXXXX",
  "accent1text": "#XXXXXX",
  "accent2": "#XXXXXX",
  "accent2text": "#XXXXXX",
  "text": "#XXXXXX",
  "headings": "#XXXXXX",
  "card": "#XXXXXX",
  "cardtext": "#XXXXXX",
  "footer": "#XXXXXX",
  "footertext": "#XXXXXX",
  "secheader": "#XXXXXX",
  "secheadertext": "#XXXXXX",
  "header": "#XXXXXX",
  "headertext": "#XXXXXX",
  "headerside": "#XXXXXX",
  "headersidetext": "#XXXXXX",
  "pagetitle": "#XXXXXX",
  "pagetitletext": "#XXXXXX",
  "graylight": "#XXXXXX",
  "graylighter": "#XXXXXX",
  "silver": "#XXXXXX",
  "body": "#XXXXXX"
}

Color explanations:
- base: Primary brand color
- basetext: Text on primary color (ensure WCAG AA contrast)
- link: Link color
- accent1/accent2: Secondary/tertiary accent colors
- accent1text/accent2text: Text on accent colors
- text: Body text color
- headings: Heading text color
- card/cardtext: Card background and text
- footer/footertext: Footer background and text
- secheader/secheadertext: Secondary header background and text
- header/headertext: Main header background and text
- headerside/headersidetext: Mobile menu background and text
- pagetitle/pagetitletext: Page title section background and text
- graylight/graylighter: Light gray utility colors
- silver: Very light background color
- body: Main page background color

Requirements:
- IMPORTANT: If the user specifies colors for specific elements (e.g., "red cards", "blue header"), you MUST use those colors for those elements
- Use 6-digit hex codes only (e.g., #1A2B3C)
- Ensure proper contrast between text and background pairs (WCAG AA minimum)
- Create a harmonious, professional color scheme
- The body background should typically be white or very light for light themes, or very dark for dark themes
PROMPT;

  /**
   * Required color keys.
   *
   * @var array
   */
  protected const REQUIRED_KEYS = [
    'base', 'basetext', 'link', 'accent1', 'accent1text', 'accent2',
    'accent2text', 'text', 'headings', 'card', 'cardtext', 'footer',
    'footertext', 'secheader', 'secheadertext', 'header', 'headertext',
    'headerside', 'headersidetext', 'pagetitle', 'pagetitletext',
    'graylight', 'graylighter', 'silver', 'body',
  ];

  /**
   * Constructs an AiPaletteGenerator object.
   *
   * @param \Drupal\ai\AiProviderPluginManager|null $ai_provider
   *   The AI provider plugin manager, or NULL if not available.
   */
  public function __construct(?AiProviderPluginManager $ai_provider = NULL) {
    $this->aiProvider = $ai_provider;
  }

  /**
   * Check if AI is available.
   *
   * @return bool
   *   TRUE if AI provider is available, FALSE otherwise.
   */
  public function isAvailable(): bool {
    return $this->aiProvider !== NULL;
  }

  /**
   * Check if a chat provider is configured and ready.
   *
   * @return bool
   *   TRUE if a chat provider is configured, FALSE otherwise.
   */
  public function hasConfiguredProvider(): bool {
    if (!$this->isAvailable()) {
      return FALSE;
    }
    // Use recommended method from AI module docs to check for configured provider.
    return $this->aiProvider->hasProvidersForOperationType('chat', TRUE);
  }

  /**
   * Generate a color palette from a prompt.
   *
   * @param string $prompt
   *   The user's prompt describing the desired palette.
   *
   * @return array
   *   An array with 'colors' on success, or 'error' on failure.
   */
  public function generate(string $prompt): array {
    if (!$this->isAvailable()) {
      return [
        'error' => $this->t('The AI module is not installed. Please install and configure the AI module to use this feature.'),
      ];
    }

    if (empty(trim($prompt))) {
      return [
        'error' => $this->t('Please enter a prompt describing your desired color palette.'),
      ];
    }

    try {
      // Check if any chat provider is configured (recommended by AI module docs).
      if (!$this->hasConfiguredProvider()) {
        return [
          'error' => $this->t('No AI chat provider configured. Please configure an AI provider in the AI module settings.'),
        ];
      }

      $default = $this->aiProvider->getDefaultProviderForOperationType('chat');
      // Verify both provider_id and model_id exist (as per AI module docs).
      if (empty($default['provider_id']) || empty($default['model_id'])) {
        return [
          'error' => $this->t('No default chat model configured. Please set a default chat model in the AI module settings.'),
        ];
      }

      $provider = $this->aiProvider->createInstance($default['provider_id']);

      $input = new ChatInput([
        new ChatMessage('user', $prompt),
      ]);
      $input->setSystemPrompt(self::SYSTEM_PROMPT);

      $response = $provider->chat($input, $default['model_id'], ['dxpr-palette']);
      $responseText = $response->getNormalized()->getText();

      // Extract JSON from response (handle potential markdown code blocks).
      $jsonText = $this->extractJson($responseText);
      $colors = json_decode($jsonText, TRUE);

      if (json_last_error() !== JSON_ERROR_NONE) {
        return [
          'error' => $this->t('Failed to parse AI response. Please try again.'),
          'debug' => $responseText,
        ];
      }

      // Validate all required color keys are present.
      $missingKeys = array_diff(self::REQUIRED_KEYS, array_keys($colors));
      if (!empty($missingKeys)) {
        return [
          'error' => $this->t('AI response missing required colors: @keys', [
            '@keys' => implode(', ', $missingKeys),
          ]),
        ];
      }

      // Validate all colors are valid hex codes.
      foreach ($colors as $key => $color) {
        if (!preg_match('/^#[0-9A-Fa-f]{6}$/', $color)) {
          return [
            'error' => $this->t('Invalid color format for @key: @color', [
              '@key' => $key,
              '@color' => $color,
            ]),
          ];
        }
      }

      return ['colors' => $colors];
    }
    catch (\Exception $e) {
      return [
        'error' => $this->t('AI request failed: @message', [
          '@message' => $e->getMessage(),
        ]),
      ];
    }
  }

  /**
   * Extract JSON from response that might be wrapped in markdown code blocks.
   *
   * @param string $text
   *   The response text.
   *
   * @return string
   *   The extracted JSON string.
   */
  protected function extractJson(string $text): string {
    // Try to extract JSON from markdown code blocks.
    if (preg_match('/```(?:json)?\s*(\{[\s\S]*?\})\s*```/', $text, $matches)) {
      return $matches[1];
    }

    // Try to find a JSON object directly.
    if (preg_match('/\{[\s\S]*\}/', $text, $matches)) {
      return $matches[0];
    }

    return $text;
  }

}
