# Dynamic Image Generator

## Introduction
The Dynamic Image Generator module creates a custom entity type for generating dynamic images using a third-party API. This module allows users to define HTML and CSS templates and then dynamically generate images based on these templates with custom token replacements.

## Features
- **Custom Entity Type**: Create and manage dynamic image templates
- **HTML/CSS Templates**: Define custom HTML and CSS for image generation
- **Token Replacement**: Support for custom tokens and automatic entity tokens
- **Background Images**: Upload and use background images in templates
- **Content Type Targeting**: Target specific content types for automatic generation
- **API Integration**: RESTful API endpoints for programmatic image generation
- **Media Integration**: Generated images are stored as Drupal media entities
- **Advanced Filtering**: Filter generated images by template name and content type
- **Preview Functionality**: Preview templates before generating final images

## Requirements
- Drupal 10.x
- Core modules: file, image, media, views
- PHP 7.4 or higher
- Active internet connection for third-party API calls

## Installation
1. Place the module in your Drupal installation's modules directory:
   ```bash
   /modules/custom/dynamic_image_generator/
   ```

2. Enable the module:
   ```bash
   drush en dynamic_image_generator
   ```
   Or go to `/admin/modules` and enable the "Dynamic Image Generator" module.

3. Clear cache:
   ```bash
   drush cr
   ```

## Configuration

### Initial Setup
1. **Configure API Settings**: Go to `/admin/config/content/dynamic-image-generator/settings` to configure API credentials and settings.

2. **Set Permissions**: Go to `/admin/people/permissions` and configure permissions:
   - `administer image template entity` - Full access to manage templates
   - `administer dynamic image generator` - Access to module configuration
   - `create image template entity` - Create new templates
   - `edit own image template entity` - Edit own templates
   - `edit any image template entity` - Edit any template
   - `delete own image template entity` - Delete own templates
   - `delete any image template entity` - Delete any template
   - `generate dynamic images` - Generate images using templates

### Template Management
1. **Manage Templates**: Go to `/admin/structure/dynamic-image-templates` to view all templates
2. **Create New Template**: Go to `/admin/structure/dynamic-image-templates/add`
3. **Template Settings**: Access general settings at `/admin/structure/dynamic-image-templates/settings`

### View Generated Images
- **Gallery View**: Visit `/config/content/dynamic-images` to see all generated images
- **Filter Options**: Use template name and content type filters
- **Admin Menu**: Access via Admin → Content → Generated Images

## Usage

### Creating Templates

#### 1. Basic Template Structure
```html
<!-- HTML Template -->
<div class="image-container" style="width: 800px; height: 600px; background-image: url('[background_image]');">
  <div class="content">
    <h1>[entity_title]</h1>
    <p class="designation">[designation]</p>
    <p class="company">[company]</p>
  </div>
</div>
```

```css
/* CSS Template */
.image-container {
  position: relative;
  background-size: cover;
  background-position: center;
  font-family: Arial, sans-serif;
}

.content {
  position: absolute;
  top: 50%;
  left: 50%;
  transform: translate(-50%, -50%);
  text-align: center;
  color: white;
  text-shadow: 2px 2px 4px rgba(0,0,0,0.7);
}

.content h1 {
  font-size: 48px;
  margin-bottom: 20px;
}

.designation {
  font-size: 24px;
  font-weight: bold;
}

.company {
  font-size: 18px;
  opacity: 0.9;
}
```

#### 2. Template Fields
- **Title**: Descriptive name for the template
- **HTML Template**: The HTML structure with token placeholders
- **CSS Template**: Styling for the HTML template
- **Background Image**: Optional background image
- **Content Type**: Target content type for automatic generation
- **Target Field**: Specific field to monitor for automatic generation
- **Status**: Enable/disable the template

### Using the Service in PHP Code

#### Basic Usage
```php
// Get the dynamic image generator service
$dynamic_image_generator = \Drupal::service('dynamic_image_generator.dynamic_image_generator_service');

// Define custom tokens to replace in the template
$tokens = [
  'designation' => 'Technical Lead',
  'company' => 'Example Corp',
  'custom_field' => 'Custom Value',
];

// Generate an image URL by passing the template entity ID and tokens
$image_url = $dynamic_image_generator->generateDynamicImage($template_entity_id, $tokens);

if ($image_url) {
  // Image generated successfully
  drupal_set_message(t('Image generated: @url', ['@url' => $image_url]));
} else {
  // Handle error
  drupal_set_message(t('Failed to generate image'), 'error');
}
```

#### Advanced Usage with Node Context
```php
// Generate image for a specific node
$node = \Drupal\node\Entity\Node::load($node_id);
$tokens = [
  'designation' => $node->get('field_designation')->value,
  'company' => $node->get('field_company')->value,
];

$image_url = $dynamic_image_generator->generateDynamicImage(
  $template_entity_id, 
  $tokens, 
  $node_id
);
```

### Using the API Endpoint

#### Generate Image via POST Request
```bash
curl -X POST \
  https://yoursite.com/api/dynamic-image-generator/{template_entity_id} \
  -H 'Content-Type: application/json' \
  -H 'X-CSRF-Token: YOUR_CSRF_TOKEN' \
  -d '{
    "designation": "Technical Lead",
    "company": "Example Corp",
    "custom_token": "Custom Value"
  }'
```

#### Response Format
```json
{
  "status": "success",
  "url": "https://yoursite.com/sites/default/files/dynamic_image_generator/generated/image_123456.png",
  "media_id": 42,
  "timestamp": "2024-01-15T10:30:00Z"
}
```

#### Error Response
```json
{
  "status": "error",
  "message": "Template not found or invalid parameters",
  "error_code": "TEMPLATE_NOT_FOUND"
}
```

### JavaScript Integration

#### Using Fetch API
```javascript
async function generateDynamicImage(templateId, tokens) {
  try {
    const response = await fetch(`/api/dynamic-image-generator/${templateId}`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-CSRF-Token': await getCsrfToken()
      },
      body: JSON.stringify(tokens)
    });
    
    const result = await response.json();
    
    if (result.status === 'success') {
      console.log('Image generated:', result.url);
      return result.url;
    } else {
      console.error('Error:', result.message);
      return null;
    }
  } catch (error) {
    console.error('Network error:', error);
    return null;
  }
}

// Usage
const imageUrl = await generateDynamicImage(123, {
  designation: 'Senior Developer',
  company: 'Tech Corp'
});
```

## Available Tokens

### Automatic Tokens
- `[background_image]` - URL of the uploaded background image (automatic)
- `[entity_title]` - Title of the template entity (automatic)
- `[entity_id]` - ID of the template entity (automatic)
- `[timestamp]` - Current timestamp (automatic)
- `[site_name]` - Site name from Drupal configuration (automatic)
- `[site_url]` - Base URL of the site (automatic)

### Node Context Tokens (when generating for a specific node)
- `[node_title]` - Title of the source node
- `[node_id]` - ID of the source node
- `[node_type]` - Content type of the source node
- `[node_author]` - Author name of the source node
- `[node_created]` - Creation date of the source node
- `[node_url]` - URL of the source node

### Custom Tokens
Any custom token can be provided when calling the generation service. Simply use the token name in square brackets in your template and provide the value in the tokens array.

## Template Examples

### Business Card Template
```html
<div class="business-card">
  <div class="header" style="background-image: url('[background_image]');">
    <h1>[full_name]</h1>
    <p class="title">[job_title]</p>
  </div>
  <div class="contact">
    <p>📧 [email]</p>
    <p>📞 [phone]</p>
    <p>🌐 [website]</p>
  </div>
  <div class="company">
    <h3>[company_name]</h3>
    <p>[company_address]</p>
  </div>
</div>
```

### Event Image Template
```html
<div class="event-image">
  <div class="background" style="background-image: url('[background_image]');"></div>
  <div class="overlay">
    <h1 class="event-title">[event_title]</h1>
    <div class="event-details">
      <p class="date">[event_date]</p>
      <p class="time">🕐 [event_time]</p>
      <p class="location">[event_location]</p>
    </div>
    <div class="description">
      <p>[event_description]</p>
    </div>
    <div class="cta">
      <p>Register at: [registration_url]</p>
    </div>
  </div>
</div>
```

### Social Media Post Template
```html
<div class="social-post">
  <div class="header">
    <img src="[profile_image]" alt="Profile" class="profile-pic">
    <div class="user-info">
      <h3>[username]</h3>
      <p>@[handle]</p>
    </div>
  </div>
  <div class="content">
    <p>[post_text]</p>
    <div class="media" style="background-image: url('[background_image]');"></div>
  </div>
  <div class="stats">
    <span>👍 [likes]</span>
    <span>💬 [comments]</span>
    <span>🔄 [shares]</span>
  </div>
</div>
```

## Third-Party API Integration

### Default API Provider
This module uses the HTML/CSS to Image API (htmlcsstoimage.com) to generate images. The API converts HTML and CSS into high-quality images.

### API Configuration
1. **API URL**: Configured in module settings
2. **Authentication**: API key and user ID
3. **Image Format**: PNG, JPEG, WebP support
4. **Quality Settings**: Configurable image quality and dimensions
5. **Timeout Settings**: Request timeout configuration

### Custom API Providers
The module supports custom API providers by implementing the image generation service interface:

```php
<?php

namespace Drupal\your_module\Plugin\ImageGenerator;

use Drupal\dynamic_image_generator\Plugin\ImageGeneratorPluginBase;

/**
 * Custom image generator plugin.
 *
 * @ImageGenerator(
 *   id = "custom_generator",
 *   label = @Translation("Custom Image Generator"),
 *   description = @Translation("Custom implementation of image generation.")
 * )
 */
class CustomImageGenerator extends ImageGeneratorPluginBase {
  
  public function generateImage($html, $css, $options = []) {
    // Custom implementation
    return $image_url;
  }
}
```

## Troubleshooting

### Common Issues

#### 1. Images Not Generating
- **Check API credentials**: Verify API settings in `/admin/config/content/dynamic-image-generator/settings`
- **Check permissions**: Ensure proper file system permissions for `/sites/default/files/dynamic_image_generator/`
- **Check logs**: Review Drupal logs at `/admin/reports/dblog`

#### 2. Template Tokens Not Replacing
- **Token syntax**: Ensure tokens use square brackets: `[token_name]`
- **Token case**: Tokens are case-sensitive
- **Token availability**: Verify tokens are passed to the generation service

#### 3. Background Images Not Loading
- **Image paths**: Ensure background images are accessible via HTTP
- **File permissions**: Check file system permissions
- **URL format**: Use absolute URLs for background images

#### 4. CSS Not Applying
- **CSS syntax**: Validate CSS syntax
- **Specificity**: Use specific selectors to override default styles
- **Inline styles**: Consider using inline styles for critical styling

### Debug Mode
Enable debug mode to get detailed error messages:

```php
// In settings.php or settings.local.php
$config['dynamic_image_generator.settings']['debug'] = TRUE;
```

### Log Analysis
Check logs for generation errors:

```bash
drush watchdog:show --filter="dynamic_image_generator"
```

## Performance Considerations

### Caching
- Generated images are cached automatically
- Template changes invalidate related image cache
- Use Drupal's cache tags for advanced cache management

### Optimization
- **Image dimensions**: Optimize template dimensions for target use case
- **CSS efficiency**: Use efficient CSS selectors and properties
- **Token minimization**: Only use necessary tokens to reduce processing

### Scaling
- **Queue integration**: Use Drupal Queue API for batch processing
- **CDN integration**: Serve generated images from CDN
- **Background processing**: Generate images asynchronously when possible

## Security Considerations

### Input Validation
- All tokens are sanitized before template processing
- HTML and CSS templates are validated for security
- File uploads are restricted to allowed image types

### Access Control
- Proper permission checks on all endpoints
- CSRF protection on API endpoints
- Access logging for audit trails

### API Security
- Secure API credential storage
- Request rate limiting
- Input size limitations

## Advanced Configuration

### Custom Token Processors
```php
<?php

/**
 * Implements hook_dynamic_image_generator_tokens_alter().
 */
function mymodule_dynamic_image_generator_tokens_alter(&$tokens, $context) {
  // Add custom tokens
  $tokens['current_year'] = date('Y');
  $tokens['user_name'] = \Drupal::currentUser()->getDisplayName();
  
  // Modify existing tokens
  if (isset($tokens['entity_title'])) {
    $tokens['entity_title'] = strtoupper($tokens['entity_title']);
  }
}
```

### Custom Image Styles
```php
<?php

/**
 * Implements hook_dynamic_image_generator_image_alter().
 */
function mymodule_dynamic_image_generator_image_alter(&$image_options, $context) {
  // Modify image generation options
  $image_options['width'] = 1200;
  $image_options['height'] = 800;
  $image_options['quality'] = 95;
  
  // Add watermark
  if ($context['template_id'] == 'public_template') {
    $image_options['watermark'] = TRUE;
  }
}
```

## API Reference

### Service Methods

#### generateDynamicImage()
```php
public function generateDynamicImage($template_id, array $tokens = [], $node_id = NULL)
```
- **Parameters**:
  - `$template_id`: Template entity ID
  - `$tokens`: Array of token replacements
  - `$node_id`: Optional node ID for context
- **Returns**: Image URL string or NULL on failure

#### getTemplateTokens()
```php
public function getTemplateTokens($template_id)
```
- **Parameters**:
  - `$template_id`: Template entity ID
- **Returns**: Array of available tokens for the template

#### validateTemplate()
```php
public function validateTemplate($html, $css)
```
- **Parameters**:
  - `$html`: HTML template string
  - `$css`: CSS template string
- **Returns**: Boolean validation result

### REST Endpoints

#### POST /api/dynamic-image-generator/{template_id}
Generate a dynamic image using the specified template.

**Request Body**:
```json
{
  "token_name": "token_value",
  "another_token": "another_value"
}
```

**Response**:
```json
{
  "status": "success|error",
  "url": "generated_image_url",
  "media_id": 123,
  "message": "status_message"
}
```

## Maintainers
- **Module Developer**: [Your Name](mailto:your.email@example.com)
- **Project Repository**: [GitHub/GitLab URL]
- **Issue Queue**: [Issue tracker URL]
- **Documentation**: [Documentation URL]

## Contributing
1. Fork the repository
2. Create a feature branch
3. Make your changes
4. Add tests for new functionality
5. Submit a pull request

## License
This module is licensed under the GPL v2 or later.

## Changelog

### Version 1.0.0
- Initial release
- Basic template creation and management
- HTML/CSS to image generation
- Token replacement system
- REST API endpoints
- Media integration
- View gallery functionality

### Version 1.1.0
- Added content type filtering
- Improved template validation
- Enhanced error handling
- Performance optimizations
- Security improvements

---

For additional support, please consult the [official documentation](URL) or contact the maintainers.