# Built-in Image Generator (Image Creating Engine)

**Fast, reliable HTML/CSS to image generation directly on your server**

## Overview

The Image Creating Engine is a Drupal module that provides local HTML/CSS to image generation using `wkhtmltoimage`. This eliminates the need for external API services and provides better performance, security, and reliability for generating dynamic images.

## Features

- **Local Generation**: No external API dependencies
- **High Performance**: Fast image generation using wkhtmltoimage
- **Reliable**: Server-optimized for consistent results
- **Secure**: All processing happens on your server
- **Cost Effective**: No per-image API costs
- **Multiple Formats**: Support for PNG and JPEG formats
- **Custom Dimensions**: Flexible width and height settings
- **CSS Support**: Full CSS styling capabilities

## Requirements

### System Requirements
- **Debian/Ubuntu Server** (tested on Debian)
- **PHP 8.0+** with shell_exec() enabled
- **Drupal 9/10**
- **wkhtmltopdf package** (includes wkhtmltoimage)

### Installation Dependencies

```bash
# Install wkhtmltopdf (includes wkhtmltoimage)
sudo apt-get update
sudo apt-get install wkhtmltopdf

# Verify installation
wkhtmltoimage --version
```

## Installation

1. **Enable the module**:
   ```bash
   drush en image_creating_engine
   ```

2. **Clear cache**:
   ```bash
   drush cache:rebuild
   ```

3. **Configure the main module** to use the built-in engine:
   - Go to `/admin/config/content/dynamic-image-generator/settings`
   - Select "Built-in Image Engine (Local)" as your provider
   - Click "Test Built-in Generator" to verify it's working

## Usage

### Basic Usage

```php
// Get the image generator service
$generator = \Drupal::service('image_creating_engine.generator');

// Generate an image
$html = '<h1 style="color: blue;">Hello World!</h1>';
$css = 'body { background: #f0f8ff; padding: 20px; font-family: Arial; }';
$options = [
  'width' => 800,
  'height' => 400,
  'format' => 'png'
];

$result = $generator->generateImage($html, $css, $options);

if ($result) {
  echo 'Image URL: ' . $result['url'];
  echo 'File URI: ' . $result['uri'];
  echo 'Generator: ' . $result['generator']; // "wkhtmltoimage"
}
```

### Available Options

```php
$options = [
  'width' => 1200,        // Image width in pixels (default: 1200)
  'height' => 630,        // Image height in pixels (default: 630)
  'format' => 'png',      // Image format: 'png' or 'jpg' (default: 'png')
  'quality' => 94,        // JPEG quality 0-100 (default: 94, only for JPEG)
];
```

### Return Value

The `generateImage()` method returns an array with:

```php
[
  'uri' => 'public://dynamic_image_generator/generated/image_123.png',
  'file' => $file_entity,           // Drupal File entity
  'url' => 'http://example.com/sites/default/files/...',
  'width' => 800,
  'height' => 400,
  'format' => 'png',
  'generator' => 'wkhtmltoimage',
  'success' => TRUE,
]
```

## Testing

### Command Line Testing

```bash
# Test the service directly
cd /var/www/bigone/dev/zi-egovreview/docroot
drush php:eval "
\$gen = \Drupal::service('image_creating_engine.generator');
\$result = \$gen->generateImage('<h1>Test</h1>', 'h1{color:red;}', []);
echo \$result ? 'SUCCESS: ' . \$result['url'] : 'FAILED';
"
```

### Manual wkhtmltoimage Testing

```bash
# Test wkhtmltoimage directly
wkhtmltoimage --width 400 --height 300 --format png \
  "data:text/html,<div style='background:blue;color:white;padding:50px;text-align:center;'>TEST</div>" \
  /tmp/test.png

# Check the generated file
ls -la /tmp/test.png
```

### Web Interface Testing

Visit `/admin/config/content/dynamic-image-generator/settings` and click **"Test Built-in Generator"** to test through the web interface.

## Troubleshooting

### Common Issues

#### 1. "wkhtmltoimage not found"
```bash
# Install wkhtmltopdf
sudo apt-get update
sudo apt-get install wkhtmltopdf

# Verify installation
which wkhtmltoimage
wkhtmltoimage --version
```

#### 2. Permission Issues
```bash
# Check /tmp permissions
ls -la /tmp
sudo chmod 1777 /tmp

# Check files directory permissions
sudo chown -R www-data:www-data /var/www/.../sites/default/files/
```

#### 3. Service Not Found
```bash
# Clear Drupal cache
drush cache:rebuild

# Check if module is enabled
drush pml | grep image_creating_engine
```

### Debug Mode

Enable debug mode in the Dynamic Image Generator settings to get detailed logging:

1. Go to `/admin/config/content/dynamic-image-generator/settings`
2. Expand "Advanced Settings"
3. Check "Enable Debug Mode"
4. Save settings

Check logs at `/admin/reports/dblog` for detailed error information.

### Log Locations

- **Drupal Logs**: `/admin/reports/dblog` (filter by "image_creating_engine")
- **System Logs**: Check `/var/log/apache2/error.log` or `/var/log/nginx/error.log`

## Performance

### Optimization Tips

1. **Use appropriate dimensions**: Smaller images generate faster
2. **Choose PNG vs JPEG**: PNG for quality, JPEG for smaller file sizes
3. **Optimize CSS**: Simpler CSS renders faster
4. **Cache results**: Store generated images to avoid regeneration

### Typical Performance

- **Small images** (400x300): ~0.5-1 seconds
- **Medium images** (800x600): ~1-2 seconds  
- **Large images** (1920x1080): ~2-4 seconds

## Comparison with External APIs

| Feature | Built-in Engine | External API |
|---------|----------------|--------------|
| **Speed** | Fast (local) | Slower (network) |
| **Cost** | Free | Per-image cost |
| **Reliability** | High | Network dependent |
| **Security** | Private | Data sent externally |
| **Dependencies** | wkhtmltopdf only | API credentials |
| **Offline** | Works offline | Requires internet |

## Technical Details

### Architecture

```
HTML + CSS Input
       ↓
InbuiltImageGenerator Service
       ↓
wkhtmltoimage Command
       ↓
PNG/JPEG Output
       ↓
Drupal File Entity
```

### File Structure

```
image_creating_engine/
├── README.md                          # This file
├── image_creating_engine.info.yml     # Module definition
├── image_creating_engine.services.yml # Service definitions
└── src/
    └── Service/
        ├── InbuiltImageGenerator.php   # Main generator service
        └── WkhtmlImageGenerator.php    # Alternative service (optional)
```

### Dependencies

- **Drupal Core**: file, system
- **PHP Extensions**: Standard (no special extensions required)
- **System**: wkhtmltopdf package

## API Reference

### InbuiltImageGenerator Service

#### Methods

##### `generateImage($html, $css, $options = [])`
Generates an image from HTML and CSS.

**Parameters:**
- `$html` (string): HTML content
- `$css` (string): CSS styles  
- `$options` (array): Generation options

**Returns:** Array with file information or NULL on failure

##### `checkDependencies()`
Checks if wkhtmltoimage is available.

**Returns:** Boolean

##### `findWkhtmlPath()`
Finds the wkhtmltoimage executable path.

**Returns:** String path or NULL

## Contributing

### Development Setup

1. Clone the repository
2. Install dependencies: `composer install`
3. Enable module: `drush en image_creating_engine`
4. Run tests: `drush php:eval "...test code..."`

### Code Standards

- Follow Drupal coding standards
- Use proper type hints
- Add comprehensive documentation
- Include error handling

## Version History

### v1.0.0 (Current)
- Initial release
- wkhtmltoimage integration
- PNG/JPEG support
- Drupal File API integration
- Comprehensive error handling
- Debug mode support

## Support

### Getting Help

1. **Check logs**: `/admin/reports/dblog`
2. **Test manually**: Use command line tools
3. **Verify dependencies**: Ensure wkhtmltopdf is installed
4. **Check permissions**: Verify file system access

### Known Limitations

- Requires server-side wkhtmltopdf installation
- Limited to wkhtmltoimage capabilities
- No JavaScript execution (by design for security)
- Server resources used for generation

## License

This module is provided under the same license as Drupal core.

---

**Happy Image Generating!**

For more information about the parent Dynamic Image Generator module, see the main module documentation.
