<?php

namespace Drupal\dynamic_image_generator;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\media\Entity\Media;

/**
 * Service for managing dynamic image media entities.
 */
class DynamicImageManager {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a DynamicImageManager object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory service.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->logger = $logger_factory->get('dynamic_image_generator');
  }

  /**
   * Get all dynamic image media entities for a specific entity.
   *
   * @param int $entity_id
   *   The entity ID.
   * @param string $entity_type
   *   The entity type (defaults to 'node').
   *
   * @return \Drupal\media\Entity\Media[]
   *   Array of dynamic image media entities.
   */
  public function getDynamicImageMediaForEntity($entity_id, $entity_type = 'node') {
    $media_storage = $this->entityTypeManager->getStorage('media');

    $query = $media_storage->getQuery()
      ->condition('bundle', 'dynamic_image')
      ->condition('field_entity_id_generated', $entity_id)
      ->accessCheck(FALSE)
      ->sort('created', 'DESC');

    $media_ids = $query->execute();

    if (empty($media_ids)) {
      return [];
    }

    return $media_storage->loadMultiple($media_ids);
  }

  /**
   * Get dynamic image media by template and entity.
   *
   * @param int $template_entity_id
   *   The template entity ID.
   * @param int $entity_id
   *   The entity ID.
   *
   * @return \Drupal\media\Entity\Media|null
   *   The dynamic image media entity or NULL if not found.
   */
  public function getDynamicImageMediaByTemplate($template_entity_id, $entity_id) {
    $media_storage = $this->entityTypeManager->getStorage('media');

    $query = $media_storage->getQuery()
      ->condition('bundle', 'dynamic_image')
      ->condition('field_poster_id', $template_entity_id)
      ->condition('field_entity_id_generated', $entity_id)
      ->accessCheck(FALSE)
      ->range(0, 1)
      ->sort('created', 'DESC');

    $media_ids = $query->execute();

    if (!empty($media_ids)) {
      $media_id = reset($media_ids);
      return $media_storage->load($media_id);
    }

    return NULL;
  }

  /**
   * Delete old dynamic image media entities for cleanup.
   *
   * @param int $entity_id
   *   The entity ID.
   * @param int $keep_count
   *   Number of recent images to keep (defaults to 5).
   *
   * @return int
   *   Number of deleted media entities.
   */
  public function cleanupOldDynamicImages($entity_id, $keep_count = 5) {
    $media_storage = $this->entityTypeManager->getStorage('media');

    // Get all dynamic image media for this entity, sorted by creation date
    $query = $media_storage->getQuery()
      ->condition('bundle', 'dynamic_image')
      ->condition('field_entity_id_generated', $entity_id)
      ->accessCheck(FALSE)
      ->sort('created', 'DESC');

    $media_ids = $query->execute();

    if (count($media_ids) <= $keep_count) {
      return 0; // Nothing to clean up
    }

    // Get the IDs of media to delete (all except the most recent $keep_count)
    $media_to_delete = array_slice($media_ids, $keep_count);

    $deleted_count = 0;
    foreach ($media_to_delete as $media_id) {
      try {
        $media = $media_storage->load($media_id);
        if ($media) {
          $media->delete();
          $deleted_count++;
        }
      }
      catch (\Exception $e) {
        $this->logger->error('Error deleting dynamic image media @id: @error', [
          '@id' => $media_id,
          '@error' => $e->getMessage(),
        ]);
      }
    }

    $this->logger->info('Cleaned up @count old dynamic image media entities for entity @entity_id', [
      '@count' => $deleted_count,
      '@entity_id' => $entity_id,
    ]);

    return $deleted_count;
  }

  /**
   * Get dynamic image media statistics.
   *
   * @return array
   *   Array containing statistics about dynamic image media.
   */
  public function getStatistics() {
    $media_storage = $this->entityTypeManager->getStorage('media');

    // Count total dynamic image media
    $total_query = $media_storage->getQuery()
      ->condition('bundle', 'dynamic_image')
      ->accessCheck(FALSE)
      ->count();
    $total_count = $total_query->execute();

    // Count dynamic image media created in the last 30 days
    $recent_query = $media_storage->getQuery()
      ->condition('bundle', 'dynamic_image')
      ->condition('created', strtotime('-30 days'), '>=')
      ->accessCheck(FALSE)
      ->count();
    $recent_count = $recent_query->execute();

    // Get template entities and their usage
    $template_storage = $this->entityTypeManager->getStorage('poster_entity');
    $template_query = $template_storage->getQuery()
      ->condition('status', 1)
      ->accessCheck(FALSE);
    $template_entities = $template_query->execute();

    $template_usage = [];
    foreach ($template_entities as $template_id) {
      $usage_query = $media_storage->getQuery()
        ->condition('bundle', 'dynamic_image')
        ->condition('field_poster_id', $template_id)
        ->accessCheck(FALSE)
        ->count();
      $usage_count = $usage_query->execute();

      $template_usage[$template_id] = $usage_count;
    }

    return [
      'total_dynamic_images' => $total_count,
      'recent_dynamic_images' => $recent_count,
      'template_usage' => $template_usage,
      'active_templates' => count($template_entities),
    ];
  }

  /**
   * Regenerate dynamic image for an entity using a specific template.
   *
   * @param int $template_entity_id
   *   The template entity ID.
   * @param int $entity_id
   *   The entity ID.
   * @param bool $force_regenerate
   *   Whether to force regeneration even if an image already exists.
   *
   * @return \Drupal\media\Entity\Media|null
   *   The created or existing media entity.
   */
  public function regenerateDynamicImage($template_entity_id, $entity_id, $force_regenerate = FALSE) {
    // Check if dynamic image already exists
    if (!$force_regenerate) {
      $existing_media = $this->getDynamicImageMediaByTemplate($template_entity_id, $entity_id);
      if ($existing_media) {
        return $existing_media;
      }
    }

    // Use the dynamic image generator service to create new image
    $dynamic_image_generator = \Drupal::service('dynamic_image_generator.dynamic_image_generator_service');
    $image_url = $dynamic_image_generator->generateDynamicImage($template_entity_id, [], $entity_id);

    if ($image_url) {
      // The service should have created the media entity
      return $this->getDynamicImageMediaByTemplate($template_entity_id, $entity_id);
    }

    return NULL;
  }

  /**
   * Get the URL of a dynamic image from media entity.
   *
   * @param \Drupal\media\Entity\Media $media
   *   The dynamic image media entity.
   * @param string $image_style
   *   Optional image style to apply.
   *
   * @return string|null
   *   The image URL or NULL if not found.
   */
  public function getDynamicImageUrl(Media $media, $image_style = NULL) {
    $image_field = $media->get('field_poster_image');
    if ($image_field->isEmpty()) {
      return NULL;
    }

    $file = $image_field->entity;
    if (!$file) {
      return NULL;
    }

    if ($image_style) {
      $image_style_storage = $this->entityTypeManager->getStorage('image_style');
      $style = $image_style_storage->load($image_style);
      if ($style) {
        return $style->buildUrl($file->getFileUri());
      }
    }

    return $file->createFileUrl();
  }
}