<?php

namespace Drupal\dynamic_image_generator\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\dynamic_image_generator\ImageTemplateInterface;
use Drupal\user\UserInterface;
use Drupal\Core\Entity\EntityStorageInterface;

/**
 * Defines the Image Template entity.
 *
 * @ContentEntityType(
 *   id = "poster_entity",
 *   label = @Translation("Image Template"),
 *   label_collection = @Translation("Image Templates"),
 *   label_singular = @Translation("image template"),
 *   label_plural = @Translation("image templates"),
 *   label_count = @PluralTranslation(
 *     singular = "@count image template",
 *     plural = "@count image templates",
 *   ),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\dynamic_image_generator\ImageTemplateListBuilder",
 *     "views_data" = "Drupal\views\EntityViewsData",
 *     "form" = {
 *       "default" = "Drupal\dynamic_image_generator\Form\ImageTemplateForm",
 *       "add" = "Drupal\dynamic_image_generator\Form\ImageTemplateForm",
 *       "edit" = "Drupal\dynamic_image_generator\Form\ImageTemplateForm",
 *       "delete" = "Drupal\dynamic_image_generator\Form\ImageTemplateDeleteForm",
 *     },
 *     "access" = "Drupal\dynamic_image_generator\ImageTemplateAccessControlHandler",
 *   },
 *   base_table = "poster_entity",
 *   admin_permission = "administer poster entity",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "title",
 *     "uuid" = "uuid",
 *     "langcode" = "langcode",
 *     "status" = "status",
 *   },
 *   links = {
 *     "canonical" = "/admin/structure/poster_entity/{poster_entity}",
 *     "add-form" = "/admin/structure/poster_entity/add",
 *     "edit-form" = "/admin/structure/poster_entity/{poster_entity}/edit",
 *     "delete-form" = "/admin/structure/poster_entity/{poster_entity}/delete",
 *     "collection" = "/admin/structure/poster_entity",
 *   },
 *   field_ui_base_route = "poster_entity.settings"
 * )
 */
class ImageTemplate extends ContentEntityBase implements ImageTemplateInterface {

  use EntityChangedTrait;

  /**
   * {@inheritdoc}
   */
  public static function preCreate(EntityStorageInterface $storage, array &$values) {
    parent::preCreate($storage, $values);
    // No user assignment needed - admin only
  }

  /**
   * {@inheritdoc}
   */
  public function getTitle() {
    return $this->get('title')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setTitle($title) {
    $this->set('title', $title);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getContentType() {
    // Check if the field exists and has a value
    if ($this->hasField('content_type') && !$this->get('content_type')->isEmpty()) {
      $value = $this->get('content_type')->getValue();
      // Handle both array structure and direct value
      if (is_array($value) && isset($value[0]['value'])) {
        return $value[0]['value'];
      }
      return $this->get('content_type')->value;
    }
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setContentType($content_type) {
    $this->set('content_type', $content_type);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getTargetField() {
    // Check if the field exists and has a value
    if ($this->hasField('target_field') && !$this->get('target_field')->isEmpty()) {
      $value = $this->get('target_field')->getValue();
      // Handle both array structure and direct value
      if (is_array($value) && isset($value[0]['value'])) {
        return $value[0]['value'];
      }
      return $this->get('target_field')->value;
    }
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setTargetField($target_field) {
    $this->set('target_field', $target_field);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getHTML() {
    return $this->get('html')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setHTML($html) {
    $this->set('html', $html);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCSS() {
    return $this->get('css')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCSS($css) {
    $this->set('css', $css);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getBackgroundImages() {
    return $this->get('background_image')->referencedEntities();
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime() {
    return $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCreatedTime($timestamp) {
    $this->set('created', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function isActive() {
    return (bool) $this->getEntityKey('status');
  }

  /**
   * {@inheritdoc}
   */
  public function setActive($active) {
    $this->set('status', $active ? TRUE : FALSE);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['title'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Template Name'))
      ->setDescription(t('The name of the Image Template.'))
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(TRUE);

    // Add content type selection field
    $fields['content_type'] = BaseFieldDefinition::create('list_string')
      ->setLabel(t('Target Content Type'))
      ->setDescription(t('Select the content type this template should be available for. Users will see auto-generation options when editing content of this type.'))
      ->setSettings([
        'allowed_values_function' => 'dynamic_image_generator_get_content_types',
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'list_default',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'options_select',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(TRUE);

    $fields['target_field'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Target Field'))
      ->setDescription(t('The field in the target content type where the generated image should be saved. This field will show auto-generation options to users.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -3.5,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -3.5,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(FALSE);

    // Changed background_image field label to "Images" and updated description
    $fields['background_image'] = BaseFieldDefinition::create('image')
      ->setLabel(t('Template Images'))
      ->setDescription(t('Images for the template. You can reference them in templates using [image_1], [image_2], etc. tokens.'))
      ->setSettings([
        'file_directory' => 'dynamic_image_generator/backgrounds',
        'alt_field' => TRUE,
        'alt_field_required' => FALSE,
        'file_extensions' => 'png jpg jpeg',
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'image',
        'weight' => -3,
      ])
      ->setDisplayOptions('form', [
        'type' => 'image_image',
        'weight' => -3,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(FALSE)
      ->setCardinality(-1); // Set unlimited cardinality to accept multiple images

    $fields['html'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('HTML Template'))
      ->setDescription(t('HTML content for the image. Use tokens like [node:title], [node:field_image], etc. for content. Use [image_1], [image_2], etc. to reference uploaded images. Supports Twig syntax.'))
      ->setSettings([
        'text_processing' => 0, // Set to 0 to disable text processing (plain text)
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -2,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textarea',
        'weight' => -2,
        'rows' => 15,
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayConfigurable('form', TRUE)
      ->setRequired(TRUE);

    $fields['css'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('CSS Template'))
      ->setDescription(t('CSS styles for the image. You can use tokens like [node:title] and [image_1], [image_2], etc. tokens in your CSS. Supports Twig syntax for conditionals, loops, etc.'))
      ->setSettings([
        'text_processing' => 0, // Set to 0 to disable text processing (plain text)
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -1,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textarea',
        'weight' => -1,
        'rows' => 15,
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayConfigurable('form', TRUE)
      ->setRequired(TRUE);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Active'))
      ->setDescription(t('A boolean indicating whether the Image Template is active.'))
      ->setDescription(t('A boolean indicating whether the Image Template is active.'))
      ->setDefaultValue(TRUE)
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'weight' => 20,
      ]);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the entity was created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the entity was last edited.'));

    return $fields;
  }
}