<?php

// File: src/Form/ImageTemplateSettingsForm.php

namespace Drupal\dynamic_image_generator\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Form for Image Template settings.
 */
class ImageTemplateSettingsForm extends FormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a new ImageTemplateSettingsForm object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'image_template_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Page header
    $form['header'] = [
      '#type' => 'markup',
      '#markup' => '<div class="template-settings-header">
        <h2>Image Template Settings</h2>
        <p>Configure settings and access all related functionality for Image Templates.</p>
      </div>',
      '#weight' => -100,
    ];

    // Quick navigation section
    $form['navigation'] = [
      '#type' => 'details',
      '#title' => $this->t('Quick Navigation'),
      '#open' => TRUE,
      '#weight' => -50,
    ];

    $form['navigation']['primary_links'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['navigation-section']],
    ];

    $form['navigation']['primary_links']['title'] = [
      '#type' => 'markup',
      '#markup' => '<h3>Template Management</h3>',
    ];

    $form['navigation']['primary_links']['links'] = [
      '#theme' => 'item_list',
      '#items' => [
        Link::createFromRoute('View All Image Templates', 'entity.poster_entity.collection', [], [
          'attributes' => ['class' => ['button', 'button--primary']],
        ]),
        Link::createFromRoute('➕ Add New Image Template', 'entity.poster_entity.add_form', [], [
          'attributes' => ['class' => ['button']],
        ]),
      ],
      '#attributes' => ['class' => ['action-links']],
    ];

    $form['navigation']['admin_links'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['navigation-section']],
    ];

    $form['navigation']['admin_links']['title'] = [
      '#type' => 'markup',
      '#markup' => '<h3>Administration</h3>',
    ];

    $form['navigation']['admin_links']['links'] = [
      '#theme' => 'item_list',
      '#items' => [
        Link::createFromRoute('Dynamic Image Generator Overview', 'dynamic_image_generator.admin', [], [
          'attributes' => ['class' => ['button']],
        ]),
        Link::createFromRoute('API Settings', 'dynamic_image_generator.settings', [], [
          'attributes' => ['class' => ['button']],
        ]),
        Link::createFromRoute('Usage Examples', 'dynamic_image_generator.example', [], [
          'attributes' => ['class' => ['button']],
        ]),
      ],
      '#attributes' => ['class' => ['action-links']],
    ];

    $form['navigation']['content_links'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['navigation-section']],
    ];

    $form['navigation']['content_links']['title'] = [
      '#type' => 'markup',
      '#markup' => '<h3>Generated Content</h3>',
    ];

    $form['navigation']['content_links']['links'] = [
      '#theme' => 'item_list',
      '#items' => [
        Link::createFromRoute('View Generated Images', 'generated_dynamic_images.page_1', [], [
          'attributes' => ['class' => ['button']],
        ]),
      ],
      '#attributes' => ['class' => ['action-links']],
    ];

    // System information section
    $form['system_info'] = [
      '#type' => 'details',
      '#title' => $this->t('System Information'),
      '#open' => FALSE,
      '#weight' => -25,
    ];

    // Count templates and generated images
    try {
      $template_count = $this->entityTypeManager
        ->getStorage('poster_entity')
        ->getQuery()
        ->accessCheck(FALSE)
        ->count()
        ->execute();

      $image_count = $this->entityTypeManager
        ->getStorage('media')
        ->getQuery()
        ->condition('bundle', 'dynamic_image')
        ->accessCheck(FALSE)
        ->count()
        ->execute();
    } catch (\Exception $e) {
      $template_count = 0;
      $image_count = 0;
    }

    $form['system_info']['stats'] = [
      '#type' => 'markup',
      '#markup' => "
        <div class='system-stats'>
          <div class='stat-item'>
            <strong>Image Templates:</strong> {$template_count}
          </div>
          <div class='stat-item'>
            <strong>Generated Images:</strong> {$image_count}
          </div>
        </div>
      ",
    ];

    // Settings content
    $form['settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Template Settings'),
      '#open' => TRUE,
      '#weight' => 0,
    ];

    $form['settings']['info'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('Image Template entities are used to create dynamic images. Each template contains HTML and CSS that can be processed to generate images via API.') . '</p>',
    ];

    // Add styling
    $form['#attached']['html_head'][] = [
      [
        '#type' => 'html_tag',
        '#tag' => 'style',
        '#value' => '
          .template-settings-header {
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
            border-left: 4px solid #28a745;
          }
          .template-settings-header h2 {
            margin: 0 0 10px 0;
            color: #2c3e50;
          }
          .navigation-section {
            margin-bottom: 20px;
          }
          .navigation-section h3 {
            margin: 0 0 10px 0;
            color: #495057;
            border-bottom: 2px solid #dee2e6;
            padding-bottom: 5px;
          }
          .action-links {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
            list-style: none;
            padding: 0;
          }
          .system-stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-top: 10px;
          }
          .stat-item {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 6px;
            border-left: 4px solid #007bff;
          }
        ',
      ],
      'template_settings_css',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // This form doesn't have any settings to save, it's just a navigation page
    $this->messenger()->addMessage($this->t('Template settings accessed.'));
  }
}