<?php

namespace Drupal\dynamic_image_generator\Controller;

use Drupal\Core\Controller\ControllerBase;

/**
 * Example controller for Poster Generator.
 */
class DynamicImageGeneratorExampleController extends ControllerBase {

  /**
   * Example page showing how to use the poster generator.
   *
   * @return array
   *   A render array for the example page.
   */
  public function examplePage() {
    $build = [];

    $build['header'] = [
      '#type' => 'markup',
      '#markup' => '<h1>Poster Generator Examples</h1><p>Learn how to create effective poster templates with real examples.</p>',
    ];

    // Social Media Example
    $build['social_media'] = [
      '#type' => 'details',
      '#title' => $this->t('📱 Social Media Post Example'),
      '#open' => TRUE,
    ];

    $build['social_media']['description'] = [
      '#type' => 'markup',
      '#markup' => '<p>Perfect for LinkedIn posts, Twitter cards, and Facebook shares.</p>',
    ];

    $build['social_media']['html'] = [
      '#type' => 'details',
      '#title' => $this->t('HTML Template'),
      '#open' => FALSE,
    ];

    $build['social_media']['html']['code'] = [
      '#type' => 'markup',
      '#markup' => '<pre><code>&lt;div class="social-poster"&gt;
  &lt;div class="background"&gt;
    &lt;img src="[image_1]" alt="Background" class="bg-image"&gt;
  &lt;/div&gt;
  &lt;div class="content"&gt;
    &lt;div class="header"&gt;
      &lt;h1&gt;[node:title]&lt;/h1&gt;
      {% if node.field_company %}
        &lt;div class="company"&gt;{{ node.field_company.value }}&lt;/div&gt;
      {% endif %}
    &lt;/div&gt;
    &lt;div class="body"&gt;
      &lt;p&gt;{{ node.body.value|striptags|slice(0, 200) }}...&lt;/p&gt;
    &lt;/div&gt;
    &lt;div class="footer"&gt;
      &lt;div class="logo"&gt;
        &lt;img src="[image_2]" alt="Logo"&gt;
      &lt;/div&gt;
      &lt;div class="date"&gt;{{ node.created.value|date("M j, Y") }}&lt;/div&gt;
    &lt;/div&gt;
  &lt;/div&gt;
&lt;/div&gt;</code></pre>',
    ];

    $build['social_media']['css'] = [
      '#type' => 'details',
      '#title' => $this->t('CSS Template'),
      '#open' => FALSE,
    ];

    $build['social_media']['css']['code'] = [
      '#type' => 'markup',
      '#markup' => '<pre><code>.social-poster {
  width: 1200px;
  height: 630px;
  position: relative;
  font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Arial, sans-serif;
  overflow: hidden;
}

.background {
  position: absolute;
  top: 0;
  left: 0;
  width: 100%;
  height: 100%;
  z-index: 1;
}

.bg-image {
  width: 100%;
  height: 100%;
  object-fit: cover;
  filter: brightness(0.7);
}

.content {
  position: relative;
  z-index: 2;
  height: 100%;
  display: flex;
  flex-direction: column;
  justify-content: space-between;
  padding: 60px;
  background: linear-gradient(45deg, rgba(0,0,0,0.3), rgba(0,0,0,0.1));
}

.header h1 {
  color: white;
  font-size: 48px;
  font-weight: 700;
  line-height: 1.2;
  margin: 0 0 10px 0;
  text-shadow: 2px 2px 4px rgba(0,0,0,0.5);
}

.company {
  color: #ffd700;
  font-size: 24px;
  font-weight: 600;
  text-transform: uppercase;
  letter-spacing: 1px;
}

.body p {
  color: white;
  font-size: 24px;
  line-height: 1.4;
  margin: 30px 0;
  text-shadow: 1px 1px 2px rgba(0,0,0,0.7);
}

.footer {
  display: flex;
  justify-content: space-between;
  align-items: center;
}

.logo img {
  height: 60px;
  width: auto;
}

.date {
  color: #ccc;
  font-size: 18px;
  font-weight: 500;
}</code></pre>',
    ];

    // Event Flyer Example
    $build['event_flyer'] = [
      '#type' => 'details',
      '#title' => $this->t('Event Flyer Example'),
      '#open' => FALSE,
    ];

    $build['event_flyer']['description'] = [
      '#type' => 'markup',
      '#markup' => '<p>Great for conferences, workshops, and meetups.</p>',
    ];

    $build['event_flyer']['html'] = [
      '#type' => 'details',
      '#title' => $this->t('HTML Template'),
      '#open' => FALSE,
    ];

    $build['event_flyer']['html']['code'] = [
      '#type' => 'markup',
      '#markup' => '<pre><code>&lt;div class="event-flyer"&gt;
  &lt;div class="header-section"&gt;
    &lt;div class="event-type"&gt;{{ node.field_event_type.value|upper }}&lt;/div&gt;
    &lt;h1&gt;[node:title]&lt;/h1&gt;
    {% if node.field_subtitle %}
      &lt;h2&gt;{{ node.field_subtitle.value }}&lt;/h2&gt;
    {% endif %}
  &lt;/div&gt;
  
  &lt;div class="details-section"&gt;
    {% if node.field_event_date %}
      &lt;div class="detail-item"&gt;
        &lt;span class="icon"&gt;&lt;/span&gt;
        &lt;span&gt;{{ node.field_event_date.value|date("F j, Y") }}&lt;/span&gt;
      &lt;/div&gt;
    {% endif %}
    
    {% if node.field_location %}
      &lt;div class="detail-item"&gt;
        &lt;span class="icon"&gt;&lt;/span&gt;
        &lt;span&gt;{{ node.field_location.value }}&lt;/span&gt;
      &lt;/div&gt;
    {% endif %}
    
    {% if node.field_price %}
      &lt;div class="detail-item price"&gt;
        &lt;span class="icon"&gt;&lt;/span&gt;
        &lt;span&gt;{{ node.field_price.value }}&lt;/span&gt;
      &lt;/div&gt;
    {% endif %}
  &lt;/div&gt;
  
  &lt;div class="image-section"&gt;
    &lt;img src="[image_1]" alt="Event Image"&gt;
  &lt;/div&gt;
  
  &lt;div class="footer-section"&gt;
    &lt;div class="website"&gt;[site:url]&lt;/div&gt;
    &lt;div class="register-cta"&gt;Register Now!&lt;/div&gt;
  &lt;/div&gt;
&lt;/div&gt;</code></pre>',
    ];

    // Product Showcase Example
    $build['product'] = [
      '#type' => 'details',
      '#title' => $this->t('Product Showcase Example'),
      '#open' => FALSE,
    ];

    $build['product']['description'] = [
      '#type' => 'markup',
      '#markup' => '<p>Perfect for e-commerce product announcements and sales.</p>',
    ];

    $build['product']['html'] = [
      '#type' => 'details',
      '#title' => $this->t('HTML Template'),
      '#open' => FALSE,
    ];

    $build['product']['html']['code'] = [
      '#type' => 'markup',
      '#markup' => '<pre><code>&lt;div class="product-showcase"&gt;
  &lt;div class="product-image"&gt;
    &lt;img src="[node:field_product_image]" alt="[node:title]"&gt;
    {% if node.field_sale_badge %}
      &lt;div class="sale-badge"&gt;{{ node.field_sale_badge.value }}&lt;/div&gt;
    {% endif %}
  &lt;/div&gt;
  
  &lt;div class="product-info"&gt;
    &lt;h1&gt;[node:title]&lt;/h1&gt;
    
    &lt;div class="price-section"&gt;
      {% if node.field_original_price %}
        &lt;span class="original-price"&gt;${{ node.field_original_price.value }}&lt;/span&gt;
      {% endif %}
      &lt;span class="sale-price"&gt;${{ node.field_price.value }}&lt;/span&gt;
    &lt;/div&gt;
    
    &lt;div class="features"&gt;
      {% for item in node.field_features %}
        &lt;div class="feature"&gt;✓ {{ item.value }}&lt;/div&gt;
      {% endfor %}
    &lt;/div&gt;
    
    &lt;div class="cta-section"&gt;
      &lt;div class="cta-button"&gt;Shop Now&lt;/div&gt;
      &lt;div class="website"&gt;[site:name]&lt;/div&gt;
    &lt;/div&gt;
  &lt;/div&gt;
  
  &lt;div class="brand-logo"&gt;
    &lt;img src="[image_1]" alt="Brand Logo"&gt;
  &lt;/div&gt;
&lt;/div&gt;</code></pre>',
    ];

    // Tips section
    $build['tips'] = [
      '#type' => 'details',
      '#title' => $this->t('Pro Tips'),
      '#open' => FALSE,
    ];

    $build['tips']['content'] = [
      '#type' => 'markup',
      '#markup' => '
        <h3>Template Best Practices:</h3>
        <ul>
          <li><strong>Fixed Dimensions:</strong> Always set fixed width/height in CSS for consistent results</li>
          <li><strong>Web-Safe Fonts:</strong> Use system fonts or web fonts that load quickly</li>
          <li><strong>High Contrast:</strong> Ensure text is readable against background images</li>
          <li><strong>Responsive Text:</strong> Use appropriate font sizes for the poster dimensions</li>
          <li><strong>Image Optimization:</strong> Use high-quality template images for best results</li>
        </ul>
        
        <h3>Token Usage:</h3>
        <ul>
          <li><code>[node:title]</code> - Content title</li>
          <li><code>[node:field_name]</code> - Custom field values</li>
          <li><code>[image_1], [image_2]</code> - Uploaded template images</li>
          <li><code>[site:name]</code> - Site name</li>
          <li><code>[node:author:name]</code> - Content author</li>
          <li><code>[node:created:custom:format]</code> - Custom date formats</li>
        </ul>
        
        <h3>Twig Features:</h3>
        <ul>
          <li><code>{% if node.field_name %}</code> - Conditional content</li>
          <li><code>{{ text|upper }}</code> - Text transformations</li>
          <li><code>{{ text|slice(0, 100) }}</code> - Text truncation</li>
          <li><code>{% for item in node.field_list %}</code> - Loop through multi-value fields</li>
          <li><code>{{ date|date("M j, Y") }}</code> - Date formatting</li>
        </ul>
      ',
    ];

    // Add CSS for better display
    $build['#attached']['html_head'][] = [
      [
        '#type' => 'html_tag',
        '#tag' => 'style',
        '#value' => '
          pre {
            background: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 6px;
            padding: 15px;
            overflow-x: auto;
            font-size: 14px;
            line-height: 1.4;
          }
          code {
            font-family: "Monaco", "Menlo", "Ubuntu Mono", monospace;
          }
          .detail-item {
            margin-bottom: 10px;
          }
          h3 {
            color: #2c3e50;
            margin-top: 25px;
          }
        ',
      ],
      'dynamic_image_generator_examples_styles',
    ];

    return $build;
  }
}