<?php

namespace Drupal\dynamic_image_generator\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for displaying generated dynamic images.
 */
class DynamicImageGeneratorViewController extends ControllerBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a DynamicImageGeneratorViewController object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * Displays the generated dynamic images page.
   *
   * @return array
   *   A render array.
   */
  public function generatedImagesPage() {
    $build = [];

    // Page header
    $build['header'] = [
      '#type' => 'markup',
      '#markup' => '<div class="generated-images-header">
        <h1>Generated Dynamic Images</h1>
        <p>Browse all automatically generated dynamic images.</p>
      </div>',
    ];

    // Try to embed the view
    try {
      $view_builder = $this->entityTypeManager->getViewBuilder('view');
      $view = \Drupal\views\Views::getView('generated_dynamic_images');

      if ($view && $view->access('page_1')) {
        $build['view'] = $view_builder->view($view, 'page_1');
      } else {
        // Fallback if view doesn't exist or no access
        $build['fallback'] = $this->buildFallbackTable();
      }
    }
    catch (\Exception $e) {
      // Fallback if view fails
      $build['fallback'] = $this->buildFallbackTable();
      \Drupal::logger('dynamic_image_generator')->error('Error loading generated images view: @error', [
        '@error' => $e->getMessage(),
      ]);
    }

    return $build;
  }

  /**
   * Build a fallback table when the view is not available.
   *
   * @return array
   *   Render array for fallback table.
   */
  protected function buildFallbackTable() {
    $build = [];

    try {
      // Get dynamic image media entities
      $media_storage = $this->entityTypeManager->getStorage('media');
      $query = $media_storage->getQuery()
        ->condition('bundle', 'dynamic_image')
        ->condition('status', 1)
        ->sort('created', 'DESC')
        ->range(0, 50)
        ->accessCheck(FALSE);

      $media_ids = $query->execute();

      if (empty($media_ids)) {
        $build['empty'] = [
          '#type' => 'markup',
          '#markup' => '<div class="empty-images-state">
            <h3>No Generated Images Found</h3>
            <p>No images have been generated yet.</p>
            <p><a href="/admin/structure/dynamic-image-templates/add" class="button button--primary">Create Image Template</a></p>
          </div>',
        ];
        return $build;
      }

      $media_entities = $media_storage->loadMultiple($media_ids);

      $rows = [];
      foreach ($media_entities as $media) {
        $created = \Drupal::service('date.formatter')->format($media->getCreatedTime(), 'medium');
        $rows[] = [
          'name' => $media->getName(),
          'created' => $created,
          'operations' => Link::fromTextAndUrl(t('View'), $media->toUrl()),
        ];
      }

      $build['table'] = [
        '#type' => 'table',
        '#header' => [
          t('Image Name'),
          t('Created'),
          t('Operations'),
        ],
        '#rows' => $rows,
        '#empty' => t('No generated images found.'),
      ];

    }
    catch (\Exception $e) {
      $build['error'] = [
        '#type' => 'markup',
        '#markup' => '<div class="error">Unable to load generated images.</div>',
      ];
    }

    return $build;
  }

}
