<?php

namespace Drupal\dynamic_placeholder\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for Dynamic Placeholder settings.
 */
class DynamicPlaceholderSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['dynamic_placeholder.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dynamic_placeholder_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('dynamic_placeholder.settings');

    $form['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Dynamic Placeholder'),
      '#description' => $this->t('Enable rotating placeholder text functionality.'),
      '#default_value' => $config->get('enabled') ?? FALSE,
    ];

    $form['target_selector'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Target Input Selector'),
      '#description' => $this->t('CSS selector for the input field. Examples: <code>input[name="search_api_fulltext"]</code>, <code>#edit-keys</code>, <code>.header-search input[type="search"]</code>'),
      '#default_value' => $config->get('target_selector') ?? '',
      '#required' => FALSE,
      '#states' => [
        'required' => [
          ':input[name="enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['placeholder_prefix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Placeholder Prefix'),
      '#description' => $this->t('Text to prepend before each placeholder item. A space will be automatically added after the prefix if not present (e.g., type "Search" and it will become "Search "). To avoid a space, leave this field empty.'),
      '#default_value' => $config->get('placeholder_prefix') !== NULL ? $config->get('placeholder_prefix') : 'Search',
      '#required' => FALSE,
      '#attributes' => [
        'placeholder' => 'Search',
      ],
      // Prevent browser from trimming the value.
      '#element_validate' => [[static::class, 'validatePrefix']],
    ];

    $form['placeholder_items'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Placeholder Items'),
      '#description' => $this->t('Enter one item per line. These will be rotated as placeholder text. Example: Curd, Books, News, Events'),
      '#default_value' => $config->get('placeholder_items') ?? '',
      '#rows' => 10,
      '#required' => FALSE,
      '#states' => [
        'required' => [
          ':input[name="enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['rotation_interval'] = [
      '#type' => 'number',
      '#title' => $this->t('Rotation Interval (milliseconds)'),
      '#description' => $this->t('Time in milliseconds between placeholder changes.'),
      '#default_value' => $config->get('rotation_interval') ?? 2000,
      '#min' => 500,
      '#max' => 60000,
      '#step' => 100,
      '#required' => TRUE,
    ];

    $form['randomize_order'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Randomize Order'),
      '#description' => $this->t('Randomize the order of placeholder items instead of cycling through them sequentially.'),
      '#default_value' => $config->get('randomize_order') ?? FALSE,
    ];

    $form['pause_on_focus'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Pause on Focus'),
      '#description' => $this->t('Pause rotation when the input field is focused. Rotation resumes when the field loses focus.'),
      '#default_value' => $config->get('pause_on_focus') ?? FALSE,
    ];

    $form['fade_effect'] = [
      '#type' => 'select',
      '#title' => $this->t('Fade Effect'),
      '#description' => $this->t('Choose the transition effect when placeholder text changes.'),
      '#options' => [
        'none' => $this->t('None (Instant)'),
        'fade' => $this->t('Fade In/Out'),
        'fade-fast' => $this->t('Fade In/Out (Fast)'),
        'fade-slow' => $this->t('Fade In/Out (Slow)'),
        'slide' => $this->t('Slide'),
      ],
      '#default_value' => $config->get('fade_effect') ?? 'none',
      '#required' => TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Custom validation for prefix to preserve trailing spaces.
   */
  public static function validatePrefix(array &$element, FormStateInterface $form_state, array &$complete_form) {
    // Get the raw value to preserve trailing spaces.
    $value = $form_state->getValue('placeholder_prefix');
    if ($value !== NULL) {
      // Store the value as-is without any trimming.
      $form_state->setValueForElement($element, $value);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $enabled = $form_state->getValue('enabled');
    $selector = trim($form_state->getValue('target_selector'));
    $items = trim($form_state->getValue('placeholder_items'));

    if ($enabled) {
      if (empty($selector)) {
        $form_state->setErrorByName('target_selector', $this->t('Target selector is required when the module is enabled.'));
      }

      if (empty($items)) {
        $form_state->setErrorByName('placeholder_items', $this->t('At least one placeholder item is required when the module is enabled.'));
      }
    }

    $interval = $form_state->getValue('rotation_interval');
    if ($interval < 500) {
      $form_state->setErrorByName('rotation_interval', $this->t('Rotation interval must be at least 500 milliseconds.'));
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('dynamic_placeholder.settings');
    
    $config->set('enabled', $form_state->getValue('enabled'));
    $config->set('target_selector', trim($form_state->getValue('target_selector')));
    
    // Get prefix value - use getValue() which preserves the raw value.
    // Don't trim prefix - preserve trailing spaces that users intentionally add.
    $prefix = $form_state->getValue('placeholder_prefix');
    // Ensure we preserve the exact value, including trailing spaces.
    if ($prefix === NULL) {
      $prefix = '';
    }
    // Store the prefix exactly as entered, with any trailing spaces.
    $config->set('placeholder_prefix', $prefix);
    
    $config->set('placeholder_items', trim($form_state->getValue('placeholder_items')));
    $config->set('rotation_interval', (int) $form_state->getValue('rotation_interval'));
    $config->set('randomize_order', $form_state->getValue('randomize_order'));
    $config->set('pause_on_focus', $form_state->getValue('pause_on_focus'));
    $config->set('fade_effect', $form_state->getValue('fade_effect'));
    
    $config->save();

    parent::submitForm($form, $form_state);
  }

}

