/**
 * @file
 * Dynamic placeholder rotation behavior.
 */

(function (Drupal, once) {
  'use strict';

  /**
   * Dynamic placeholder behavior.
   */
  Drupal.behaviors.dynamicPlaceholder = {
    attach: function (context, settings) {
      // Check if settings are available.
      if (!settings.dynamicPlaceholder) {
        return;
      }

      const config = settings.dynamicPlaceholder;
      const selector = config.selector;
      // Get prefix exactly as provided - preserve trailing spaces.
      let prefix = config.prefix !== undefined ? String(config.prefix) : '';
      // Ensure prefix ends with a space if it's not empty and doesn't already end with whitespace.
      // This is a workaround for browsers/form processing that might trim trailing spaces.
      if (prefix && prefix.length > 0 && !/\s$/.test(prefix)) {
        prefix = prefix + ' ';
      }
      const items = config.items || [];
      const interval = config.interval || 2000;
      const randomize = config.randomize || false;
      const pauseOnFocus = config.pauseOnFocus || false;
      const fadeEffect = config.fadeEffect || 'none';

      // If no items or selector, do nothing.
      if (!items.length || !selector) {
        return;
      }

      // Find all matching inputs using once() to ensure we only process each once.
      // once() returns a NodeList, convert to array for easier manipulation.
      const inputs = Array.from(once('dynamic-placeholder', selector, context));
      
      if (!inputs || inputs.length === 0) {
        return;
      }

      // Prepare items array.
      let itemsArray = [...items];
      if (randomize) {
        // Shuffle array using Fisher-Yates algorithm.
        for (let i = itemsArray.length - 1; i > 0; i--) {
          const j = Math.floor(Math.random() * (i + 1));
          [itemsArray[i], itemsArray[j]] = [itemsArray[j], itemsArray[i]];
        }
      }

      // Initialize for each input.
      inputs.forEach(function (input) {
        let currentIndex = 0;
        let rotationTimer = null;
        let isPaused = false;
        const originalPlaceholder = input.placeholder || '';

        // Apply fade effect class if needed.
        if (fadeEffect !== 'none') {
          input.classList.add('dynamic-placeholder-' + fadeEffect);
        }

        /**
         * Update placeholder text with fade effect.
         */
        function updatePlaceholder() {
          // Don't update if input has a value (user has typed something).
          if (input.value && input.value.trim() !== '') {
            return;
          }

          // Don't update if paused.
          if (isPaused) {
            return;
          }

          // Get current item.
          const currentItem = itemsArray[currentIndex];
          const newPlaceholder = prefix + currentItem;

          // Apply fade effect if enabled.
          if (fadeEffect !== 'none') {
            // Determine fade duration based on effect type.
            const fadeOutDuration = fadeEffect === 'fade-fast' ? 75 : fadeEffect === 'fade-slow' ? 250 : 150;
            const fadeInDuration = fadeEffect === 'fade-fast' ? 75 : fadeEffect === 'fade-slow' ? 250 : 150;
            
            // Fade out by reducing opacity.
            input.classList.add('dynamic-placeholder-' + fadeEffect + '-out');
            
            // Wait for fade out, then change text and fade in.
            setTimeout(function() {
              input.placeholder = newPlaceholder;
              // Force reflow to ensure transition works.
              void input.offsetWidth;
              // Remove fade-out class to fade back in.
              input.classList.remove('dynamic-placeholder-' + fadeEffect + '-out');
            }, fadeOutDuration);
          } else {
            // No fade effect - instant change.
            input.placeholder = newPlaceholder;
          }

          // Move to next item.
          if (randomize) {
            // For randomize, pick a random item each time.
            currentIndex = Math.floor(Math.random() * itemsArray.length);
          } else {
            // For sequential, move to next item.
            currentIndex = (currentIndex + 1) % itemsArray.length;
          }
        }

        /**
         * Start rotation.
         */
        function startRotation() {
          // Clear any existing timer.
          if (rotationTimer) {
            clearInterval(rotationTimer);
          }

          // Set initial placeholder.
          updatePlaceholder();

          // Start interval.
          rotationTimer = setInterval(updatePlaceholder, interval);
        }

        /**
         * Stop rotation.
         */
        function stopRotation() {
          if (rotationTimer) {
            clearInterval(rotationTimer);
            rotationTimer = null;
          }
        }

        /**
         * Handle focus event.
         */
        function handleFocus() {
          if (pauseOnFocus) {
            isPaused = true;
            stopRotation();
          }
        }

        /**
         * Handle blur event.
         */
        function handleBlur() {
          if (pauseOnFocus) {
            isPaused = false;
            // Only resume if input is empty.
            if (!input.value || input.value.trim() === '') {
              startRotation();
            }
          }
        }

        /**
         * Handle input event (user typing).
         */
        function handleInput() {
          // If user clears the input, restore rotation.
          if (!input.value || input.value.trim() === '') {
            if (!isPaused) {
              startRotation();
            }
          }
        }

        // Attach event listeners.
        input.addEventListener('focus', handleFocus);
        input.addEventListener('blur', handleBlur);
        input.addEventListener('input', handleInput);

        // Start rotation.
        startRotation();
      });
    }
  };

})(Drupal, once);

