<?php

namespace Drupal\dynamic_selection_tools\Plugin\Field\FieldWidget;

use Drupal\Core\Form\FormStateInterface;
use Drupal\dynamic_selection_tools\Plugin\Field\DynamicOptionsWidget;

/**
 * Plugin implementation of the 'numeric_range_options' widget.
 *
 * @FieldWidget(
 *   id = "numeric_range_options",
 *   module = "dynamic_selection_tools",
 *   label = @Translation("Numeric range options"),
 *   field_types = {
 *     "string",
 *     "integer",
 *     "float",
 *     "decimal"
 *   }
 * )
 */
class NumericRangeOptionsWidget extends DynamicOptionsWidget {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return parent::defaultSettings() + [
      'numeric_range' => [
        'step' => 1,
        'leading_zero' => TRUE,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $settings = $this->getSettings();
    $element = parent::settingsForm($form, $form_state);
    $element['common']['options']['#description'] = $this->t('Enter your desired options range in the format OFFSET|LIMIT. For example @range_example.', ['@range_example' => '0|9']);
    $element['numeric_range'] = [
      '#type' => 'container',
      '#tree' => TRUE,
      '#weight' => -49,
    ];
    $element['numeric_range']['step'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Step'),
      '#default_value' => $settings['numeric_range']['step'],
      '#required' => TRUE,
    ];
    $element['numeric_range']['leading_zero'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use leading zeros for lower value options'),
      '#default_value' => $settings['numeric_range']['leading_zero'],
      '#description' => $this->t('NOTE: This will work only for <strong>integer</strong> values.'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function provideOptions(array &$form, FormStateInterface $form_state) : array {
    $items = [];
    $settings = $this->getSettings();
    $entity = $form_state->getFormObject()->getEntity();
    $options = $this->replaceTokens($settings['common']['options'], [$entity->getEntityTypeId() => $entity]);
    $range = explode('|', $options);
    $step = $this->replaceTokens($settings['numeric_range']['step'], [$entity->getEntityTypeId() => $entity]);
    if (
      count($range) == 2 &&
      isset($range[0]) &&
      isset($range[1])
    ) {
      $offset = trim(strip_tags($range[0]));
      $limit = trim(strip_tags($range[1]));
      $step = trim(strip_tags($step));
      $length = $format = NULL;
      if (
        $settings['numeric_range']['leading_zero'] &&
        ($offset === (string) (int) $offset) &&
        ($limit === (string) (int) $limit) &&
        ($step === (string) (int) $step)
      ) {
        // Leading zeros is for integer values. So, that's reason for this.
        $length = strlen($limit);
        $format = "%0{$length}d";
      }
      if (is_numeric($offset) && is_numeric($limit) && is_numeric($step)) {
        for ($i = $offset; $i <= $limit; $i += $step) {
          $option = $format ? sprintf($format, $i) : $i;
          $items[(string) $option] = $option;
        }
      }
    }

    return $items;
  }

}
