<?php

namespace Drupal\dynamic_selection_tools\Service;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Psr\Log\LoggerInterface;

/**
 * Handle text pad features.
 */
class TextPad implements TextPadInterface {

  protected LoggerInterface $logger;

  /**
   * Class constructor.
   *
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(LoggerChannelFactoryInterface $logger_factory) {
    $this->logger = $logger_factory->get('dynamic_selection_tools');
  }

  /**
   * {@inheritdoc}
   */
  public function padEntityFieldValue(ContentEntityInterface $entity, string $field_name, int $delta, string $field_property, string $character, int $amount, string $direction = 'left') {
    $text = '';
    if (!$entity->hasField($field_name)) {
      return $text;
    }
    $items = $entity->get($field_name);
    if (!isset($items[$delta])) {
      return $text;
    }
    $item = $items[$delta];
    $value = (string) ($item->{$field_property} ?: '');
    if (!$value) {
      return $text;
    }
    $text = $this->padText($value, $character, $amount, $direction);

    return $text;
  }

  /**
   * {@inheritdoc}
   */
  public function padText(string $text, string $character, int $amount, string $direction) {
    $output = '';
    if ($text && $character && $amount > 0 && in_array($direction, ['left', 'right', 'both'])) {
      if (strlen($character) !== 1) {
        $character = substr($character, 0, 1);
      }
      if ($amount < strlen($text)) {
        $this->logger->warning('Padding amount (@amount) smaller than text length (@length). Returning original text: @text', [
          '@amount' => $amount,
          '@length' => strlen($text),
          '@text' => $text,
        ]);

        return $text;
      }
      switch ($direction) {
        case 'left':
          $output = str_pad($text, $amount, $character, STR_PAD_LEFT);
          break;

        case 'right':
          $output = str_pad($text, $amount, $character, STR_PAD_RIGHT);
          break;

        case 'both':
          $output = str_pad($text, $amount, $character, STR_PAD_BOTH);
          break;
      }
    }
    else {
      $this->logger->warning('Failed to pad text with values: Text @text; Amount @amount; Character @character Direction @direction.', [
        '@text' => $text,
        '@amount' => $amount,
        '@character' => $character,
        '@direction' => $direction,
      ]);
    }

    return $output ?: $text;
  }

}
