<?php

namespace Drupal\dynamic_selection_tools\Plugin\Field;

use Drupal\Component\Utility\Html;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\FocusFirstCommand;
use Drupal\Core\Ajax\InsertCommand;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Render\AttachmentsInterface;
use Drupal\Core\Render\Element;
use Drupal\Core\Utility\Token;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base field for dynamic options widget.
 */
class DynamicOptionsWidget extends WidgetBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The token service.
   *
   * @var \Drupal\Core\Utility\Token
   */
  protected $token;

  /**
   * {@inheritdoc}
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, array $third_party_settings, EntityTypeManagerInterface $etm, Token $token) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
    $this->entityTypeManager = $etm;
    $this->token = $token;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('entity_type.manager'),
      $container->get('token')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'common' => [
        'options' => '',
        'sort' => '_none',
        'counter' => '',
        'theme' => TRUE,
        'option_add_text' => '',
        'option_remove_text' => '',
      ],
      'advanced' => [
        'single_value_selection' => FALSE,
        'storage_validation' => FALSE,
        'storage_conditions' => '',
      ],
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $settings = $this->getSettings();
    $field_name = $this->fieldDefinition->getName();
    $element['common'] = [
      '#type' => 'container',
      '#tree' => TRUE,
      '#weight' => -50,
    ];
    $element['common']['options'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Options'),
      '#default_value' => $settings['common']['options'],
      '#required' => TRUE,
    ];
    $element['common']['sort'] = [
      '#type' => 'select',
      '#title' => $this->t('Sort'),
      '#options' => [
        '_none' => $this->t('- None -'),
        'ASCENDING' => $this->t('Ascending'),
        'DESCENDING' => $this->t('Descending'),
      ],
      '#default_value' => $settings['common']['sort'],
    ];
    $element['common']['counter'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Counter selector(s)'),
      '#default_value' => $settings['common']['counter'],
      '#description' => $this->t('Common HTML tags and input fields are supported, but for input fields be sure you provide the tag name as part of the selector. For example <em>@input_example</em>', ['@input_example' => '.parent-container .form-item-textfield input']),
    ];
    $element['common']['theme'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Provide basic theming'),
      '#default_value' => $settings['common']['theme'],
    ];
    $element['common']['option_add_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Override selection button text'),
      '#default_value' => $settings['common']['option_add_text'],
    ];
    $element['common']['option_remove_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Override remove button text'),
      '#default_value' => $settings['common']['option_remove_text'],
    ];
    $element['advanced'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced'),
      '#tree' => TRUE,
      '#weight' => 50,
    ];
    $element['advanced']['single_value_selection'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Single value selection'),
      '#default_value' => $settings['advanced']['single_value_selection'],
    ];
    $element['advanced']['storage_validation'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Storage validation'),
      '#default_value' => $settings['advanced']['storage_validation'],
    ];
    $_sch = [
      '#type' => 'inline_template',
      '#template' => '
        <div class="guidelines intro">{{text_intro}}</div>
        <div class="examples" style="background-color: #ebebeb; color: #000000; padding: 3px;">
          <div class="example">field_my_field\FOO\=;</div>
          <div class="example">field_another_field\BAR\!=;</div>
          <div class="example">field_entity_reference.entity:node.field_in_referenced_entity\VALUE0,VALUE1,VALUE2\IN</div>
        </div>
        <div class="guidelines token">{{text_token}}</div>
      ',
      '#context' => [
        'text_intro' => $this->t(
          '
            WARNING: Use this feature with caution. If you enter a wrong line, you may to break your site. Enter one condition per line,
            according to describen in <a href=":external_docs" target="_blank">:external_docs</a> and separating each condition component by "\" (forward slash)
            and each condition by ";" (semicolon). For example:
          ',
          [
            ':external_docs' => 'https://www.drupal.org/docs/8/api/database-api/dynamic-queries/conditions',
          ]
        ),
        'text_token' => $this->t('This field also supports usage of Tokens. So, be sure you use correct values.'),
      ],
    ];
    $element['advanced']['storage_conditions'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Aditional storage conditions'),
      '#default_value' => $settings['advanced']['storage_conditions'],
      '#description' => static::renderer()->renderPlain($_sch),
      '#states' => [
        'visible' => [
          ':input[name="fields[' . $field_name . '][settings_edit_form][settings][advanced][storage_validation]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $element['token_browser'] = [
      '#theme' => 'token_tree_link',
      '#token_types' => [
        $form['#entity_type'],
      ],
      '#weight' => 100,
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $settings = $this->getSettings();
    $summary = [
      $this->t('Options: @options', ['@options' => $settings['common']['options']]),
      $this->t('Sort: @sort', ['@sort' => $settings['common']['sort']]),
      $this->t('Counter: @counter', ['@counter' => $settings['common']['counter']]),
      $this->t('Theme: @theme', ['@theme' => $settings['common']['theme'] ? $this->t('Yes') : $this->t('No')]),
      $this->t('Option add text: @op_add_text', ['@op_add_text' => $settings['common']['option_add_text']]),
      $this->t('Option remove text: @op_remove_text', ['@op_remove_text' => $settings['common']['option_remove_text']]),
    ];

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function form(FieldItemListInterface $items, array &$form, FormStateInterface $form_state, $get_delta = NULL) {
    $elements = parent::form($items, $form, $form_state, $get_delta);
    $settings = $this->getSettings();
    if (isset($settings['common']['theme']) && (bool) $settings['common']['theme']) {
      $elements['#attached']['library'][] = 'dynamic_selection_tools/dst.widget';
    }

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  protected function formMultipleElements(FieldItemListInterface $items, array &$form, FormStateInterface $form_state) {
    $elements = parent::formMultipleElements($items, $form, $form_state);
    $settings = $this->getSettings();
    $field_name = $this->fieldDefinition->getName();
    $entity = $form_state->getFormObject()->getEntity();
    $elements['#attributes']['class'][] = 'dst--container';
    // We're going to replace the default "Add item" button. But
    // we need some of its property values before doing that.
    // @TODO: Maybe this is not necessary. Check and fix.
    $id_prefix = implode('-', array_merge($form['#parents'], [$field_name]));
    if (isset($elements['add_more']['#ajax']['wrapper'])) {
      $wrapper = $elements['add_more']['#ajax']['wrapper'];
    }
    else {
      $wrapper = Html::getUniqueId($id_prefix . '-add--more-wrapper');
      $elements['#prefix'] = '<div id="' . $wrapper . '">';
      $elements['#suffix'] = '</div>';
    }
    $elements['add_more'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['dst--add_more', 'dst--add_more--' . $field_name],
      ],
    ];
    $options = $this->getOptions($form, $form_state);
    $field_state = $this->getFieldState($form, $form_state);
    if (!empty($options)) {
      $_oc = 0;
      $replacements = [
        $entity->getEntityTypeId() => $entity,
      ];
      $add_option_text = $this->replaceTokens($settings['common']['option_add_text'] ?? '', $replacements, [], $this->t('Add'));
      $remove_option_text = $this->replaceTokens($settings['common']['option_remove_text'] ?? '', $replacements, [], $this->t('Remove'));
      $records = $this->getRecords($form_state, array_keys($options));
      foreach ($options as $value => $option) {
        $disabled = FALSE;
        $taken = isset($records[$value]);
        $picked = in_array($value, $field_state['selected_items']);
        $option_selector = 'dst--add_more--item--value--' . $_oc . '--' . $wrapper;
        if (
          (
            isset($settings['advanced']['single_value_selection']) &&
            (bool) $settings['advanced']['single_value_selection'] &&
            in_array($value, $field_state['selected_items'])
          )
          ||
          (
            $elements['#cardinality'] != FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED &&
            $field_state['items_count'] >= $elements['#cardinality']
          )
          ||
          (
            isset($records[$value])
          )
        ) {
          $disabled = TRUE;
        }
        $elements['add_more']['add_more_' . $_oc] = [
          '#type' => 'container',
          '#attributes' => [
            'class' => [
              'dst--add_more--item',
              'dst--add_more--item-' . $_oc,
              $disabled ? 'disabled' : 'enabled',
              $taken ? 'busy' : 'free',
              $picked ? 'picked' : '',
              $_oc % 2 == 0 ? 'odd' : 'even',
            ],
          ],
        ];
        $elements['add_more']['add_more_' . $_oc]['option'] = [
          '#theme' => 'dynamic_options__option',
          '#option_label' => $option,
          '#option_value' => $value,
          '#context' => [
            'is_disabled' => $disabled,
            'is_taken' => $taken,
            'is_picked' => $picked,
            'entity' => $entity,
            'records' => isset($records[$value]) ? $records[$value] : [],
          ],
        ];
        $elements['add_more']['add_more_' . $_oc]['value'] = [
          '#type' => 'submit',
          '#name' => str_replace('-', '_', $id_prefix) . "__{$_oc}_add_button",
          '#op' => 'add_item',
          '#container' => "#$wrapper",
          '#selector' => ".$option_selector",
          '#text' => $value,
          '#value' => $add_option_text,
          '#limit_validation_errors' => [],
          '#submit' => [[$this, 'addItemSubmit']],
          '#ajax' => [
            'callback' => [$this, 'addItemAjaxCallback'],
            'wrapper' => $wrapper,
            'effect' => 'fade',
          ],
          '#attributes' => [
            'class' => [
              'dst--add_more--item--value',
              $option_selector,
            ],
            'disabled' => $disabled,
          ],
        ];
        $_oc++;
      }
      // Override the "Remove" ajax callback button,
      // in order to use our own one with extra logic.
      foreach (Element::children($elements) as $delta) {
        if (is_numeric($delta)) {
          if (isset($elements[$delta]['_actions']['delete'])) {
            $elements[$delta]['_actions']['delete']['#op'] = 'remove_item';
            $elements[$delta]['_actions']['delete']['#container'] = "#$wrapper";
            $elements[$delta]['_actions']['delete']['#submit'] = [[$this, 'removeItemSubmit']];
            $elements[$delta]['_actions']['delete']['#ajax']['callback'] = [$this, 'removeItemAjaxCallback'];
            $elements[$delta]['_actions']['delete']['#value'] = $remove_option_text;
          }
          else {
            $elements[$delta]['_actions'] = [
              'delete' => [
                '#type' => 'submit',
                '#name' => str_replace('-', '_', $id_prefix) . "__{$delta}_remove_button",
                '#delta' => $delta,
                '#op' => 'remove_item',
                '#container' => "#$wrapper",
                '#value' => $remove_option_text,
                '#validate' => [],
                '#submit' => [[$this, 'removeItemSubmit']],
                '#limit_validation_errors' => [],
                '#ajax' => [
                  'callback' => [$this, 'removeItemAjaxCallback'],
                  'wrapper' => $wrapper,
                  'effect' => 'fade',
                ],
              ],
              '#weight' => 101,
            ];
          }
        }
      }
    }

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $value = $items[$delta]->value ?? NULL;
    $element['value'] = $element + [
      '#type' => 'textfield',
      '#default_value' => $value,
      '#maxlength' => $this->getFieldSetting('max_length'),
      '#element_validate' => [[$this, 'validateSelectedItem']],
      '#attributes' => [
        'readonly' => 'readonly',
      ],
    ];

    return $element;
  }

  /**
   * Override submit function to add item.
   *
   * @param array $form
   *   The field form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current $form_state.
   */
  public function addItemSubmit(array $form, FormStateInterface $form_state) {
    $button = $form_state->getTriggeringElement();
    $element = NestedArray::getValue($form, array_slice($button['#array_parents'], 0, -3));
    $field_name = $element['#field_name'];
    $parents = $element['#field_parents'];
    // Increment the items count.
    $field_state = $this->getFieldState($form, $form_state);
    if (($element['#cardinality'] === FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED) || ($field_state['items_count'] < $element['#cardinality'])) {
      $field_state['items_count']++;
      array_push($field_state['selected_items'], $button['#text']);
    }
    else {
      if ($element['#cardinality'] === 1) {
        $field_state['items_count'] = 1;
        $field_state['selected_items'] = [$button['#text']];
      }
    }
    static::setWidgetState($parents, $field_name, $form_state, $field_state);
    $form_state->setRebuild();
  }

  /**
   * A last stand: Validate selected items on submit.
   */
  public function validateSelectedItem($element, FormStateInterface $form_state) {
    $value = $element['#value'];
    if (strlen($value) == 0) {
      $form_state->setValueForElement($element, '');

      return;
    }
    $records = $this->getRecords($form_state, [$value]);
    if (isset($records[$value])) {
      $form_state->setError($element, $this->t('Value "@value" is already taken. Please, pick another option.', ['@value' => $value]));
    }
  }

  /**
   * Override parent add ajax callback to add extra settings.
   *
   * @param array $form
   *   The field form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current $form_state.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The AJAX response object.
   */
  public function addItemAjaxCallback(array $form, FormStateInterface $form_state) {
    $button = $form_state->getTriggeringElement();
    $elements = NestedArray::getValue($form, array_slice($button['#array_parents'], 0, -3));
    $value = $button['#text'];
    $field_state = $this->getFieldState($form, $form_state);
    $delta = $field_state['items_count'] <= 1 ? 0 : $field_state['items_count'] - 1;
    $elements[$delta]['#prefix'] = '<div class="ajax-new-content">' . ($elements[$delta]['#prefix'] ?? '');
    $elements[$delta]['#suffix'] = ($elements[$delta]['#suffix'] ?? '') . '</div>';
    $elements[$delta]['value']['#value'] = $value;
    // Turn render array into response with AJAX commands.
    $response = new AjaxResponse();
    $response->addCommand(new InsertCommand(NULL, $elements));
    $response->addCommand(new FocusFirstCommand($button['#selector']));
    $context = [
      'op' => 'add_item',
      'entity' => $form_state->getFormObject()->getEntity(),
      'field_state' => $field_state,
    ];
    $this->doAlterAjaxResponse($response, $context);

    return $response;
  }

  /**
   * Override submit function removal.
   *
   * @param array $form
   *   The field form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current $form_state.
   */
  public function removeItemSubmit(array &$form, FormStateInterface $form_state) {
    static::deleteSubmit($form, $form_state);
    $field_state = $this->getFieldState($form, $form_state);
    $deleted_item = $field_state['deleted_item'] ?? NULL;
    if (!is_null($deleted_item)) {
      unset($field_state['selected_items'][$deleted_item]);
      $field_state['selected_items'] = array_values($field_state['selected_items']);
      static::setWidgetState($form['#parents'], $this->fieldDefinition->getName(), $form_state, $field_state);
    }
    $form_state->setRebuild();
  }

  /**
   * Override parent delete ajax callback to add extra settings.
   *
   * @param array $form
   *   The field form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current $form_state.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The AJAX response object.
   */
  public function removeItemAjaxCallback(array &$form, FormStateInterface $form_state) {
    $element = static::deleteAjax($form, $form_state);
    $response = new AjaxResponse();
    $response->addCommand(new InsertCommand(NULL, $element));
    $context = [
      'op' => 'remove_item',
      'entity' => $form_state->getFormObject()->getEntity(),
      'field_state' => $this->getFieldState($form, $form_state),
    ];
    $this->doAlterAjaxResponse($response, $context);

    return $response;
  }

  /**
   * Helper function to easily get field state.
   *
   * @param array $form
   *   The field form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current $form_state.
   *
   * @return array
   *   The current field state.
   */
  public function getFieldState(array &$form, FormStateInterface $form_state) {
    $field_name = $this->fieldDefinition->getName();
    $parents = $form['#parents'];
    $field_state = array_merge(['selected_items' => []], static::getWidgetState($parents, $field_name, $form_state));

    return $field_state;
  }

  /**
   * Wrapper function to add initial ajax response settings.
   *
   * @param \Drupal\Core\Render\AttachmentsInterface $response
   *   The returning response object.
   * @param array $context
   *   Arbitrary data for this request.
   */
  private function doAlterAjaxResponse(AttachmentsInterface &$response, array &$context = []) {
    if (
      isset($context['op']) &&
      in_array($context['op'], ['add_item', 'remove_item']) &&
      isset($context['entity']) &&
      $context['entity'] instanceof ContentEntityInterface
    ) {
      $settings = $this->getSettings();
      if (
        isset($settings['common']['counter']) &&
        !empty($settings['common']['counter']) &&
        isset($context['field_state']['items_count'])
      ) {
        $counter = $this->replaceTokens($settings['common']['counter'], [$context['entity']->getEntityTypeId() => $context['entity']]);
        $selectors = explode(',', $counter);
        foreach ($selectors as $selector) {
          $matches = [];
          preg_match('/input|select|textarea/', $selector, $matches);
          $method = empty($matches) ? 'text' : 'val';
          $response->addCommand(new InvokeCommand($selector, $method, [$context['field_state']['items_count']]));
          if ($method == 'val') {
            $response->addCommand(new InvokeCommand($selector, 'trigger', ['change']));
          }
        }
      }
    }
    $this->alterAjaxResponse($response, $context);
  }

  /**
   * Helper function to alter the add/remove ajax response.
   *
   * @param \Drupal\Core\Render\AttachmentsInterface $response
   *   The returning response object.
   * @param array $context
   *   Arbitrary data for this request.
   */
  public function alterAjaxResponse(AttachmentsInterface &$response, array &$context = []) {}

  /**
   * Load existing entities for a given set of options.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current $form_state.
   * @param array $allowed_values
   *   Allowed values to check.
   *
   * @return array
   *   The loaded entities, keyed by specified allowed values.
   */
  public function getRecords(FormStateInterface $form_state, array $allowed_values) : array {
    $records = [];
    $settings = $this->getSettings();
    $field_name = $this->fieldDefinition->getName();
    $entity = $form_state->getFormObject()->getEntity();
    if (!(bool) $settings['advanced']['storage_validation'] || empty($allowed_values) || !$entity->hasField($field_name)) {
      return $records;
    }
    $entity_storage = $this->entityTypeManager->getStorage($entity->getEntityTypeId());
    $query = $entity_storage->getQuery()
      ->accessCheck(FALSE)
      ->condition($field_name, $allowed_values, 'IN');
    $conditions = $this->replaceTokens($settings['advanced']['storage_conditions'] ?? '', [$entity->getEntityTypeId() => $entity]);
    if (!empty($conditions)) {
      $conditions = explode(';', $conditions);
      foreach ($conditions as $condition) {
        $fragments = explode('\\', $condition);
        $_field_name = $fragments[0] ?? NULL;
        $_value = $fragments[1] ?? NULL;
        $_operator = $fragments[2] ?? NULL;
        if ($_field_name && $_value && $_operator) {
          if (in_array($_operator, ['IN', 'NOT IN'])) {
            $_value = explode(',', $_value);
          }
          $query->condition($_field_name, $_value, $_operator);
        }
      }
    }
    $result = $query->execute();
    // Let's "normalize" each value with its respective set of entities,
    // for easier validation.
    if (!empty($result)) {
      $_entities = $entity_storage->loadMultiple($result);
      foreach ($_entities as $_entity) {
        foreach ($_entity->get($field_name)->getValue() as $_ev) {
          $records[$_ev['value']][] = $_entity;
        }
      }
    }

    return $records;
  }

  /**
   * Overridable function to provide an associative array of options.
   *
   * Return an associative array in the way of VALUE => LABEL.
   *
   * @param array $form
   *   The form field.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form field state.
   *
   * @return array
   *   The array of options.
   */
  public function provideOptions(array &$form, FormStateInterface $form_state) : array {
    return [];
  }

  /**
   * Function to get an associative array of options.
   *
   * @param array $form
   *   The form field.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form field state.
   *
   * @return array
   *   The array of options.
   */
  protected function getOptions(array &$form, FormStateInterface $form_state) : array {
    $options = $this->provideOptions($form, $form_state);
    if (!empty($options)) {
      $settings = $this->getSettings();
      switch ($settings['common']['sort']) {
        case 'ASCENDING':
          asort($options);
          break;

        case 'DESCENDING':
          arsort($options);
          break;
      }
    }

    return $options;
  }

  /**
   * Replaces tokens into the provided string.
   *
   * @param string $value
   *   The string to be processed.
   * @param array $replacements
   *   An array of keyed entities.
   * @param array $options
   *   Extra settings for this replacement.
   * @param string $default_value
   *   Default value in case the resulting value is empty.
   *
   * @return string
   *   Value with tokens replaced.
   */
  protected function replaceTokens(string $value, array $replacements, array $options = [], string $default_value = '') {
    if (!empty($value)) {
      $_options = [
        'clear' => TRUE,
      ] + $options;
      $value = $this->token->replace($value, $replacements, $_options);
    }

    return empty($value) && !empty($default_value) ? $default_value : $value;
  }

  /**
   * Get renderer service for arbitrary markup displays.
   */
  protected static function renderer() {
    return \Drupal::service('renderer');
  }

}
