<?php

namespace Drupal\dynamic_selection_tools\Plugin\Field\FieldWidget;

use Drupal\Core\Field\Plugin\Field\FieldWidget\EntityReferenceAutocompleteWidget;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines an Entity Reference Autocomplete widget with query string parameters.
 *
 * @FieldWidget(
 *   id = "entity_reference_autocomplete_query_string_params",
 *   label = @Translation("Autocomplete with query string parameters"),
 *   field_types = {
 *     "entity_reference"
 *   }
 * )
 */
class EntityReferenceAutocompleteWithQueryStringParametersWidget extends EntityReferenceAutocompleteWidget implements ContainerFactoryPluginInterface {

  /**
   * The request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The token service.
   *
   * @var \Drupal\Core\Utility\Token
   */
  protected $token;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->requestStack = $container->get('request_stack');
    $instance->token = $container->get('token');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return parent::defaultSettings() + [
      'query_string_items' => '',
      'availability' => 'include',
      'custom_query_string_items' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = parent::settingsForm($form, $form_state);
    $elements['query_string_items'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Query string items'),
      '#description' => $this->t('Enter a list of query string items, separated by the "&" (ampersand) character. For example, <em>@example</em>.', ['@example' => 'item&option&amount']),
      '#default_value' => $this->getSetting('query_string_items'),
    ];
    $elements['availability'] = [
      '#type' => 'select',
      '#title' => $this->t('Availability'),
      '#description' => $this->t('Determine if previous query string items should be included or excluded.'),
      '#options' => [
        'include' => $this->t('Include'),
        'exclude' => $this->t('Exclude'),
      ],
      '#default_value' => $this->getSetting('availability'),
    ];
    $elements['custom_query_string_items'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Custom query string items'),
      '#description' => $this->t('Enter a list of query string items and their respective values, separated by the "&" (ampersand) character. This setting supports Tokens usage. For example, <em>@example</em>.', ['@example' => 'item=[node:nid]&option=custom&amount=1']),
      '#default_value' => $this->getSetting('custom_query_string_items'),
    ];
    $elements['token_browser'] = [
      '#theme' => 'token_tree_link',
      '#token_types' => [
        $form['#entity_type'],
      ],
      '#weight' => 100,
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $element = parent::formElement($items, $delta, $element, $form, $form_state);
    $query = $this->requestStack->getCurrentRequest()->query->all();
    if (count($query) == 0) {
      return $element;
    }
    $settings = $this->getWidgetSettings();
    if (count($settings['query_string_items']) > 0) {
      switch ($settings['availability']) {
        case 'include':
          foreach ($query as $key => $_) {
            if (!in_array($key, $settings['query_string_items'])) {
              unset($query[$key]);
            }
          }
          break;

        case 'exclude':
          foreach ($query as $key => $_) {
            if (in_array($key, $settings['query_string_items'])) {
              unset($query[$key]);
            }
          }
          break;
      }
    }
    if (count($settings['custom_query_string_items']) > 0) {
      $entity = $form_state->getFormObject()->getEntity();
      $replacements = [
        $entity->getEntityTypeId() => $entity,
      ];
      foreach ($settings['custom_query_string_items'] as $_q) {
        $_tmp = $this->replaceTokens($_q, $replacements);
        if (!empty($_tmp)) {
          [$_qs, $_qv] = array_pad(explode('=', $_tmp), 2, NULL);
          if ($_qs && $_qv) {
            $query[$_qs] = $_qv;
          }
        }
      }
    }
    if (!empty($query)) {
      $element['target_id']['#autocomplete_query_parameters'] = $query;
    }

    return $element;
  }

  /**
   * Get formatted settings.
   */
  public function getWidgetSettings() : array {
    $settings = $this->getSettings();
    $settings['query_string_items'] = !empty($settings['query_string_items']) ? explode('&', $settings['query_string_items']) : [];
    $settings['custom_query_string_items'] = !empty($settings['custom_query_string_items']) ? explode('&', $settings['custom_query_string_items']) : [];

    return $settings;
  }

  /**
   * Replaces tokens into the provided string.
   *
   * @param string $value
   *   The string to be processed.
   * @param array $replacements
   *   An array of keyed entities.
   * @param array $options
   *   Extra settings for this replacement.
   * @param string $default_value
   *   Default value in case the resulting value is empty.
   *
   * @return string
   *   Value with tokens replaced.
   */
  protected function replaceTokens(string $value, array $replacements, array $options = [], string $default_value = '') {
    if (!empty($value)) {
      $_options = [
        'clear' => TRUE,
      ] + $options;
      $value = $this->token->replacePlain($value, $replacements, $_options);
    }

    return empty($value) && !empty($default_value) ? $default_value : $value;
  }

}
