<?php

namespace Drupal\dynamic_selection_tools\Plugin\views\argument_default;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheableDependencyInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Plugin\views\argument_default\ArgumentDefaultPluginBase;
use Drupal\user\EntityOwnerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The current entity owner ID.
 *
 * @ingroup views_argument_default_plugins
 *
 * @ViewsArgumentDefault(
 *   id = "entity_owner",
 *   title = @Translation("Owner ID from entity in current route context")
 * )
 * @package Drupal\dynamic_selection_tools\Plugin\views\argument_default
 */
class EntityOwnerArgument extends ArgumentDefaultPluginBase implements CacheableDependencyInterface {

  /**
   * The route match service.
   *
   * @var \Drupal\Core\Routing\CurrentRouteMatch
   */
  protected $routeMatch;

  /**
   * The token service.
   *
   * @var \Drupal\Core\Utility\Token
   */
  protected $token;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create(
      $container,
      $configuration,
      $plugin_id,
      $plugin_definition
    );
    $instance->token = $container->get('token');
    $instance->routeMatch = $container->get('current_route_match');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['argument'] = ['default' => ''];
    $options['fallback_value'] = ['default' => ''];

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    $form['argument'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Route argument name'),
      '#default_value' => $this->options['argument'],
    ];
    $form['fallback_value'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Fallback value'),
      '#description' => $this->t('Value in case the route argument is not met. This also supports usage of basic tokens.'),
      '#default_value' => $this->options['fallback_value'],
    ];
    $form['tokens'] = [
      '#theme' => 'token_tree_link',
      '#global_types' => TRUE,
      '#click_insert' => TRUE,
      '#show_restricted' => FALSE,
      '#recursion_limit' => 3,
      '#text' => $this->t('Browse available tokens'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getArgument() {
    $argument = NULL;
    $route_argument = $this->options['argument'];
    $fallback_value = $this->options['fallback_value'];
    $fallback_argument = $this->token->replace($fallback_value, [], ['clear' => TRUE]);
    if (!empty($route_argument)) {
      $entity = $this->routeMatch->getParameter(trim($route_argument));
      if ($entity instanceof EntityOwnerInterface) {
        $argument = $entity->getOwner()->id();
      }
    }

    return $argument ? $argument : $fallback_argument;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheMaxAge() {
    return Cache::PERMANENT;
  }

  /**
   * {@inheritdoc}
   *
   * @TODO Define proper cache contexts.
   */
  public function getCacheContexts() {
    return [];
  }

}
