<?php

namespace Drupal\eaf;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides base plugin implementation for entity attribute plugins.
 *
 * @package Drupal\eaf
 */
abstract class EntityAttributePluginBase extends PluginBase implements EntityAttributePluginInterface, ContainerFactoryPluginInterface {

  /**
   * Entity attributes plugin manager.
   *
   * @var \Drupal\eaf\EntityAttributePluginManagerInterface
   */
  protected $attributePluginManager;

  /**
   * Constructs entity plugin instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\eaf\EntityAttributePluginManagerInterface $attribute_manager
   *   Entity attributes plugin manager.
   *
   * @todo Add constructor property promotion
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    EntityAttributePluginManagerInterface $attribute_manager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->attributePluginManager = $attribute_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): self {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('eaf.eaf_plugin_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getSettings(): array {
    return $this->pluginDefinition['settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(): array {
    return [
      '#title' => $this->label(),
      "#default_value" => $this->getDefaultValue(),
      '#weight' => $this->getWeight(),
      '#plugin' => $this,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function id(): string {
    return $this->getPluginId();
  }

  /**
   * {@inheritdoc}
   */
  public function label(): ?TranslatableMarkup {
    return $this->pluginDefinition['label'] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultValue(): mixed {
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getParentPluginId(): ?string {
    return $this->getSettings()['parent'] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function hasParentPlugin(): bool {
    return is_string($this->getParentPluginId());
  }

  /**
   * {@inheritdoc}
   */
  public function isPluginParent(string $parent_id): bool {
    return $this->getParentPluginId() === $parent_id;
  }

  /**
   * {@inheritdoc}
   */
  public function isSubtype(): bool {
    return $this->hasParentPlugin();
  }

  /**
   * {@inheritdoc}
   */
  public function getWeight(): int {
    return $this->getSettings()['weight'] ?? 0;
  }

  /**
   * {@inheritdoc}
   */
  public function getValidationRulesDefinition(): array {
    return [];
  }

  /**
   * Getter for css class list.
   *
   * @param mixed $value
   *   Stored value.
   *
   * @return array
   *   Array of classes.
   */
  protected function getElementList($value): array {
    if (is_array($value)) {
      return $value;
    }

    if (is_string($value) || is_null($value)) {
      return $this->getListFromString($value);
    }

    return [];
  }

  /**
   * Get CSS class from a given string.
   *
   * @param null|string $value
   *   String to be parsed.
   *
   * @return array
   *   CSS classes.
   */
  protected function getListFromString(?string $value): array {
    $classes = preg_split('/[\s,]+/', $value, -1, PREG_SPLIT_NO_EMPTY);

    return $classes ?: [];
  }

  /**
   * {@inheritdoc}
   */
  public function prepareFormElement(array $element, mixed $value): array {
    $type = $element['#subtype'] ?? $element['#type'];

    // @todo move these type handlers out of own methods?
    switch ($type) {
      case 'css':
      case 'element-list':
        $classes = $this->getElementList($value);
        // $subtype = EntitySettingPluginManagerInterface::SETTINGS_SUBTYPE_ID;
        // $this->processModifiers($element, $options[$subtype], $classes);
        //
        // // Preserve only root keys, without children/modifiers.
        // unset($options[$subtype]);
        $new_element = [
          '#default_value' => implode(' ', $classes),
        ] + $element;
        break;

      case 'select':
        $new_element = [
          '#default_value' => empty($value) ? $element['#default_value'] : $value,
        ] + $element;
        break;

      case 'number':
        $new_element = [
          '#default_value' => !empty($value) && $value !== '' ? $value : $element['#default_value'],
          '#min' => $element['#min'] ?? NULL,
          '#max' => $element['#max'] ?? NULL,
        ] + $element;
        break;

      case 'checkboxes':
        // Convert stdClass to array after deserialization.
        $value = is_object($value) ? (array) $value : $value;
        $new_element = [
          '#default_value' => empty($value) ? $element['#default_value'] : $value,
        ] + $element;
        break;

      default:
        $value = $value ?? $element['#default_value'];

        $new_element = [
          '#size' => $element['#size'] ?? 32,
          '#default_value' => $value,
        ] + $element;
        break;
    }

    return $new_element;
  }

  /**
   * Set the plugin value to the field storage.
   *
   * @param mixed $value
   *   The plugin value.
   * @param array $attributes
   *   The attributes array.
   * @param string $section
   *   The field value's section (the field name).
   *
   * @return array
   *   The attributes array.
   */
  public function setValue(
    mixed $value,
    array $attributes,
    string $section = NULL,
  ): array {

    if ($section) {
      $attributes[$section][$this->id()] = $value;
    }
    else {
      $attributes[$this->id()] = $value;
    }
    return $attributes;
  }

  /**
   * @todo should Add addValue()?
   */
  // public function addValue(
  //   mixed $new_value,
  //   array $attributes,
  //   string $section = NULL,
  // ) {
  //   // @todo
  // }

}
