<?php

namespace Drupal\eaf;

use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Provides interface for entity attributes plugins.
 *
 * @package Drupal\eaf
 */
interface EntityAttributePluginInterface {

  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    EntityAttributePluginManagerInterface $attribute_manager,
  );

  /**
   * Plugin settings.
   *
   * @return array
   *   List of settings.
   */
  public function getSettings(): array;

  /**
   * Entity attribute form element.
   *
   * This is a main settings component used, to build the form
   * containing all of the settings.
   *
   * @return array
   *   Form element.
   */
  public function formElement(): array;

  /**
   * Prepare the plugin form element.
   *
   * @param $element
   *   The form element options.
   * @param $value
   *   The form element value.
   *
   * @return array
   *   The prepared form element.
   */
  public function prepareFormElement(array $element, array $value): array;

  /**
   * Getter for plugin id.
   *
   * @return string
   *   The plugin id.
   */
  public function id(): string;

  /**
   * Getter for plugin label.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup|null
   *   The plugin label.
   */
  public function label(): ?TranslatableMarkup;

  /**
   * Getter for form element #default_value.
   *
   * @return mixed
   *   Value to be used as a default
   */
  public function getDefaultValue(): mixed;

  /**
   * Set (add) the plugin value to the field storage.
   *
   * @param mixed $value
   *   The new value.
   * @param array $attributes
   *   The attributes array.
   *
   * @return array
   *   The updated attributes array.
   */
  public function setValue(mixed $value, array $attributes): array;

  /**
   * Getter for parent plugin id.
   *
   * @return string|null
   *   The plugin id or null if this is a root element.
   */
  public function getParentPluginId(): ?string;

  /**
   * Check if the plugin has parent.
   *
   * @return bool
   *   True if has parent, false if this is a root element.
   */
  public function hasParentPlugin(): bool;

  /**
   * Check if the given plugin is the plugin parent.
   *
   * @param string $parent_id
   *   Id of plugin parent to be checked.
   *
   * @return bool
   *   True if this is the plugin parent, false otherwise.
   */
  public function isPluginParent(string $parent_id): bool;

  /**
   * This is an alias of hasParentPlugin.
   *
   * @return bool
   *   True if has parent, false if this is a root element.
   */
  public function isSubtype(): bool;

  /**
   * Getter for plugin weight, used as #weight in form element.
   *
   * @return int
   *   Plugin weight.
   */
  public function getWeight(): int;

  /**
   * Getter for validation rules definition.
   *
   * @return array
   *   Validation rules.
   */
  public function getValidationRulesDefinition(): array;

}
