<?php

namespace Drupal\eaf;

use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Plugin\DefaultPluginManager;

/**
 * The plugin manager for entity attribute plugins.
 *
 * @package Drupal\eaf
 */
class EntityAttributePluginManager extends DefaultPluginManager implements EntityAttributePluginManagerInterface {

  use LoggerChannelTrait;

  /**
   * Logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Creates the discovery object.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   Cache backend instance to use.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler to invoke the alter hook with.
   */
  public function __construct(\Traversable $namespaces,
                              CacheBackendInterface $cache_backend,
                              ModuleHandlerInterface $module_handler) {
    parent::__construct(
      'Plugin/EntityAttribute',
      $namespaces,
      $module_handler,
      'Drupal\eaf\EntityAttributePluginInterface',
      'Drupal\eaf\Attribute\EntityAttribute'
    );

    $this->alterInfo('entity_setting_info');
    $this->setCacheBackend($cache_backend, 'entity_attribute_plugins');
    $this->logger = $this->getLogger('eaf');
  }

  /**
   * {@inheritdoc}
   */
  public function getAll(): array {
    return $this->loadPluginsFromDefinitions($this->getDefinitions());
  }

  /**
   * {@inheritdoc}
   */
  public function getPluginById(string $plugin_id) {
    return $this->createInstance($plugin_id);
  }

  /**
   * @return array
   *
   * @deprecated
   * @see \Drupal\eaf\EntityAttributePluginManager::getAttributeFormElements()
   */
  public function getSettingsFieldForm(): array {
    return $this->getAttributeFormElements();
  }

  /**
   * {@inheritdoc}
   *
   * @todo Rename it! plugin id = $plugin->formElement()
   */
  public function getAttributeFormElements(): array {
    $cache = $this->cacheGet(self::SETTINGS_FORM_STORAGE_CID);
    // @todo Comment out the next line to enable the cache:
    // $cache = NULL;

    if ($cache) {
      $form = $cache->data;
    }
    else {
      /** @var \Drupal\eaf\EntityAttributePluginInterface[] $plugins */
      $plugins = $this->getAll();
      $form = [];

      foreach ($plugins as $plugin) {
        $form[$plugin->id()] = $plugin->formElement();
      }
      $this->moduleHandler->alter('entity_attributes', $form);

      $this->cacheSet(self::SETTINGS_FORM_STORAGE_CID,
        $form,
        Cache::PERMANENT,
        [self::SETTINGS_FORM_STORAGE_CID]
      );
    }

    return $form;
  }

  /**
   * Load all plugins by given definitions.
   *
   * @param array $definitions
   *   Plugin definitions.
   *
   * @return array
   *   Loaded plugin instances.
   */
  protected function loadPluginsFromDefinitions(array $definitions): array {
    $plugins = [];

    foreach ($definitions as $definition) {
      try {
        // @todo We can think of keeping the configuration in yml files.
        $plugins[$definition['id']] = $this->getPluginById($definition['id']);
      }
      catch (PluginException $exception) {
        $this->logger->error($exception->getMessage());
      }
    }

    return $plugins;
  }

}
