<?php

namespace Drupal\eaf;

use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Core\Field\FieldItemList;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\TypedData\DataDefinitionInterface;
use Drupal\Core\TypedData\TypedDataInterface;

/**
 * Provides field item list class for attribute storage field type.
 *
 * @package Drupal\eaf
 */
class EntityAttributesFieldItemList extends FieldItemList implements EntityAttributesFieldItemListInterface {

  use LoggerChannelTrait;

  /**
   * Logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Entity attributes plugin manager.
   *
   * @var \Drupal\eaf\EntityAttributePluginManagerInterface
   */
  protected $attributePluginManager;

  /**
   * {@inheritdoc}
   */
  public function __construct(DataDefinitionInterface $definition, $name = NULL, TypedDataInterface $parent = NULL) {
    parent::__construct($definition, $name, $parent);

    $this->attributePluginManager = \Drupal::service('eaf.eaf_plugin_manager');
    $this->logger = $this->getLogger('eaf');
  }

  /**
   * {@inheritdoc}
   */
  public function getFieldAttributeValue(string $target_section = NULL): array {
    if (empty($this->list[0])) {
      return [];
    }
    $values = $this->list[0]->getFieldAttributeValue($target_section);
    return $values ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function getFieldItemAttributeValue(string $target_section = NULL): array {
    if (empty($this->list[0])) {
      return [];
    }
    $values = $this->list[0]->getFieldItemAttributeValue($target_section);
    return $values ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function getEntityAttributeValue(string $target_section = NULL): array {
    $values = [];

    foreach ($this->list as $delta => $item) {
      $values[$delta] = $item->getEntityAttributeValue($target_section);
    }

    return $values[0] ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function setFieldAttributeValue($values, string $target_section = NULL, $notify = TRUE) {
    foreach ($this->list as $delta => $item) {
      $item->setFieldAttributeValue($values, $target_section);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function defaultValuesFormSubmit(array $element, array &$form, FormStateInterface $form_state) {
    $value = parent::defaultValuesFormSubmit($element, $form, $form_state);

    return $this->toEncodedValue($value);
  }

  /**
   * {@inheritdoc}
   */
  public function hasSettingValue(string $setting_name): bool {
    return !is_null($this->getSettingValue($setting_name));
  }

  /**
   * {@inheritdoc}
   */
  public function getSettingValue(string $setting_name, $default = NULL) {
    return $this->getValue()->$setting_name ?? $this->getStorageItemDefaultValue($setting_name) ?? $default;
  }

  /**
   * {@inheritdoc}
   */
  public function setSettingValue(string $name, $value): EntityAttributesFieldItemListInterface {
    $values = $this->getValue();
    $values->$name = $value;

    $this->setEncodedValue($values);

    return $this;
  }

  /**
   * Set value as JSON encoded string.
   *
   * @param mixed $values
   *   The values to be set.
   *
   * @throws \Drupal\Core\TypedData\Exception\ReadOnlyException
   */
  protected function setEncodedValue($values) {
    $this->setValue($this->toEncodedValue($values), TRUE);
  }

  /**
   * Convert given values to a field value with encoded data.
   *
   * @param mixed $value
   *   The values to be set.
   *
   * @return array[]
   *   The field value.
   */
  protected function toEncodedValue($value): array {
    return [
      [
        'value' => json_encode($value),
      ],
    ];
  }

  /**
   * Getter for storage item default value.
   *
   * @param string $storage_id
   *   The storage id.
   *
   * @return mixed|null
   *   The default value.
   */
  protected function getStorageItemDefaultValue(string $storage_id) {
    try {
      return $this->loadStorageItemById($storage_id)->getDefaultValue();
    }
    catch (PluginException $exception) {
      return NULL;
    }
  }

  /**
   * Loads storage item.
   *
   * @param string $storage_id
   *   The storage id (plugin id).
   *
   * @return \Drupal\eaf\EntityAttributePluginInterface|object
   *   The plugin instance.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  protected function loadStorageItemById(string $storage_id) {
    return $this->attributePluginManager->getPluginById($storage_id);
  }

}
