<?php

namespace Drupal\eaf\Plugin\Field\FieldWidget;

use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eaf\EntityAttributePluginManagerInterface;
use Drupal\eaf\EntityAttributes;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'Entity attributes field' widget.
 */
#[FieldWidget(
  id: 'eaf_widget',
  label: new TranslatableMarkup('Entity attributes widget'),
  field_types: [
    'field_attributes_storage'
  ],
  multiple_values: FALSE
)]
class AttributesFieldWidget extends WidgetBase {

  /**
   * The entity attributes plugin manager.
   *
   * @var \Drupal\eaf\EntityAttributePluginManagerInterface
   */
  protected $attributePluginManager;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): self {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->attributePluginManager = $container->get('eaf.eaf_plugin_manager');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array {
    $summary = [];
    $summary[] = $this->t('This is an attributes field.');

    return $summary;
  }

  /**
   * Get widget forms for attribute plugins.
   *
   * @return array
   *   The all plugin widgets.
   */
  private function getPluginWidgets(): array {
    $widgets = $this->attributePluginManager->getAttributeFormElements();
    return $widgets;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(
    FieldItemListInterface $items,
    $delta,
    array $element,
    array &$form,
    FormStateInterface $form_state,
  ): array {
    $field_value = $items->getEntityAttributeValue();

    /** @var \Drupal\eaf\Plugin\Field\FieldType\AttributesStorageItem $field */
    $field = $items[$delta];
    $field_settings = $field->getFieldDefinition()
      ->getSetting('entity_attribute_plugins');
    $plugin_widgets = $this->getPluginWidgets();
    $widgets = [];
    // Config options = form elements.
    foreach ($plugin_widgets as $key => $options) {
      if (empty($field_settings[$key])) {
        // Filter out the not enabled elements.
        continue;
      }
      /** @var \Drupal\eaf\EntityAttributePluginInterface $plugin */
      $plugin = $options['#plugin'];
      $value = $field_value[$key] ?? '';

      $widgets[$key] = $plugin->prepareFormElement($options, $value);
    }

    if ($widgets) {
      // Set up the form element for this widget.
      $label = $field->getFieldDefinition()->getLabel();
      $element['#title'] = $this->t('@label (content)', ['@label' => $label]);
      $element += [
        '#type' => 'details',
        '#open' => FALSE,
      ];

      $element += $widgets;

      $element['#attached']['library'][] = 'eaf/eaf.attribute-widget';
    }
    // Add validate hook.
    $element += [
      '#element_validate' => [
        [$this, 'validate'],
      ],
    ];
    return [
      'value' => $element,
    ];
  }

  /**
   * Validate the fields and convert them into a single value as json.
   *
   * {@inheritdoc}
   */
  public function validate(array $element, FormStateInterface $form_state): void {
    $entity_attributes = [
      'classes' => [],
    ];
    $field_item_attributes = [];
    $field_attributes = [];

    $field_name = $this->fieldDefinition->getName();
    $plugin_widgets = $this->getPluginWidgets();
    $settings = $this->fieldDefinition->getSettings();

    // The entity attributes part.
    foreach ($plugin_widgets as $key => $options) {
      $value = $form_state->getValue(array_merge($element['#parents'], [$key]));
      /** @var \Drupal\eaf\EntityAttributePluginInterface $plugin */
      $plugin = $options['#plugin'];

      $entity_attributes = $plugin->setValue($value, $entity_attributes);
    }

    $index = array_search($field_name, $element['#parents']);
    $parents = array_slice($element['#parents'], 0, $index);
    $field_item_settings = $settings['field_item_attribute_plugins'] ?? [];

    $section = EntityAttributes::FIELD_ITEM_ATTRIBUTES;
    foreach ($field_item_settings as $f_name => $plugins) {
      foreach ($plugins as $id => $status) {
        if (empty($status)) {
          // Filter out the not enabled elements.
          continue;
        }
        // $plugin = $plugin_widgets[$id]['#plugin'];
        $submitted = $form_state->getValue(array_merge($parents, [$f_name])) ?? [];
        // @todo: The plugin setValue() sets the global "class" also.
        $attribute_values = [
          // @todo Classes handling.
          'classes' => [],
        ];

        foreach ($submitted as $delta => $value) {
          if (is_int($delta) && !empty($value[$section])
            && is_array($value[$section])) {
            $attribute_values[$delta] = $value[$section] ?? [];
          }
        }

        $field_item_attributes[$f_name] = $attribute_values;
      }
    }

    $field_settings = $settings['field_attribute_plugins'] ?? [];
    $section = EntityAttributes::FIELD_ATTRIBUTES;
    foreach ($field_settings as $f_name => $plugins) {
      foreach ($plugins as $id => $status) {
        if (empty($status)) {
          // Filter out the not enabled elements.
          continue;
        }
        $submitted = $form_state->getValue(array_merge($parents, [$f_name])) ?? [];
        $attribute_values = [
          'classes' => [],
        ];

        // For the field attributes the delta is the "add_more" now.
        $delta = 'add_more';
        $value = $submitted[$delta] ?? [];
        if (!empty($value[$section])
          && is_array($value[$section])) {
          $attribute_values += $value[$section];
        }
        $field_attributes[$f_name] = $attribute_values;
      }
    }

    $entity_values = [
      EntityAttributes::ENTITY_ATTRIBUTES => $entity_attributes,
      EntityAttributes::FIELD_ITEM_ATTRIBUTES => $field_item_attributes,
      EntityAttributes::FIELD_ATTRIBUTES => $field_attributes,
    ];

    $form_state->setValueForElement($element, json_encode($entity_values));
  }

  /**
   * Getter for target bundle of the entity.
   *
   * @return string|null
   *   Bundle name, or null if it's not bundle specific.
   */
  protected function getTargetBundle(): ?string {
    return $this->fieldDefinition->getTargetBundle();
  }

}
